# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from django.core.urlresolvers import reverse

from common.utils.locations import langify
from common.xgettext.i18n import gettext, xgettext

from ..base import Base
from ..utils import wrap_with_link_item
from ..blocks import ErrorDisclaimerBlock


class Template(Base):
    page = 'search-result-on-name-station'
    show_filters = True

    def input(self):
        form = self.context.form

        point_name = form['pointName'].data
        point_key = form['pointId'].data

        if point_name:
            point_name = point_name

        if point_key:
            point_key = point_key

        return {
            'block': 'b-wrapper',
            'attrs': {'action': langify(reverse('station_search')), 'method': 'get'},
            'mods': {'type': 'search-input', 'margin': 'big'},
            'content': {
                'block': 'b-search',
                'content': [
                    {
                        'elem': 'input',
                        'content': {
                            'block': 'input',
                            'js': {
                                'dataprovider': {
                                    'name': 'dataprovider',
                                    'suggestsType': 'station'
                                },
                                'name': 'pointName'
                            },
                            'mods': {'suggest': 'yes', 'pick': 'yes', 'size': 'm', 'theme': 'city'},
                            'content': [
                                {
                                    'elem': 'label',
                                    'attrs': {'for': 'input-search-trip'},
                                    'content': gettext('Вы искали')
                                },
                                {
                                    'elem': 'control',
                                    'attrs': {
                                        'id': 'input-search-trips',
                                        'placeholder': gettext('Название станции'),
                                        'name': 'pointName',
                                        'value': point_name
                                    }
                                },
                                {
                                    'elem': 'key',
                                    'tag': 'input',
                                    'attrs': {'name': 'pointId', 'value': point_key, 'type': 'hidden'}
                                }
                            ]
                        }
                    },
                    {
                        'elem': 'button',
                        'content': {
                            'block': 'button',
                            'mods': {'theme': 'search', 'size': 'xl'},
                            'attrs': {'type': 'submit', 'role': 'submit'},
                            'content': gettext('Найти')
                        }
                    }
                ]
            }
        }

    def station_filter(self):
        return None

        some_from_current_country = any(variant for header, variants in self.context.variants
                                        for variant in variants
                                        if self.point_filter_value(variant) == 'current')
        some_from_another_country = any(variant for header, variants in self.context.variants
                                        for variant in variants
                                        if self.point_filter_value(variant) == 'other')

        if not some_from_another_country or not some_from_current_country:
            return None

        items = [
            {
                'js': {'value': 'all'},
                'text': gettext('Все')
            },
            {
                'js': {'value': 'current'},
                'text': self.context.current_country_name
            },
            {
                'js': {'value': 'other'},
                'text': gettext('За рубежом')
            },
        ]

        return {
            'block': 'b-wrapper',
            'tag': 'div',
            'mods': {'type': 'search-switcher'},
            'content': SwitcherTemplate(value='all',
                                        items=items,
                                        mods={'type': 'station-filter'})
        }

    def content(self):
        return [
            {
                'elem': 'item',
                'content': [
                    self.input(),
                    self.station_filter()
                ]
            },
            {
                'elem': 'item',
                'content': self.precise_content()
            }
        ]

    def point_filter_value(self, point):
        country_id = self.context.form.country_id
        country_key = 'l' + unicode(country_id)

        if getattr(point, 'country_id', None) == country_id or point.point_key == country_key:
            return 'current'

        return 'other'

    def point_place(self, point):
        if getattr(point, 'has_omonims', None):
            national_version = self.context.request.NATIONAL_VERSION
            show_district = national_version == 'ru'
            return point.L_full_geography(show_district=show_district, national_version=national_version)

        if getattr(point, 'settlement', None):
            return point.settlement.L_title_with_prefix()

        if getattr(point, 'region', None):
            return point.region.L_title()

        return point.country.L_title()

    def precise_content(self):
        form = self.context.form

        if not form.is_valid():
            return [ErrorDisclaimerBlock(self.context.error)]

        variant_groups = self.context.variants

        if not variant_groups:
            return [ErrorDisclaimerBlock(gettext('К сожалению, у нас нет информации о станции с таким названием.'))]

        content = []

        for header, variants in variant_groups:
            content.extend([
                {
                    'block': 'b-title',
                    'mods': {'size': 'small', 'filter': self.point_filter_value(variants[0])},
                    'js': {'filter_value': self.point_filter_value(variants[0])},
                    'content': header
                },
                {
                    'block': 'b-routers',
                    'js': {'filter_value': self.point_filter_value(variants[0])},
                    'elemMods': {'type': 'search-result', 'filter': self.point_filter_value(variants[0])},
                    'content': [{
                        'elem': 'line',
                        'js': {'filter_value': self.point_filter_value(variant)},
                        'mix': [{'block': 'i-pressed-controller'}],
                        'content': [{
                            'elem': 'item',
                            'content': [
                                {
                                    'block': 'b-link',
                                    'url': variant.url,
                                    'content': [{
                                        'elem': 'item',
                                        'content': variant.L_title_with_type()
                                    }]
                                },
                                {
                                    'elem': 'location',
                                    'content': wrap_with_link_item(self.point_place(variant))
                                }
                            ]
                        }]
                    } for variant in variants]
                }
            ])

        return content


class Country(Template):
    page = 'search-result-on-name-station'
    show_filters = False

    def precise_content(self):
        city_airports = self.context.airports

        content = [{
            'block': 'b-title',
            'mods': {'size': 'small'},
            'content': self.context.precise_title
        }]

        for settlement, stations in city_airports:
            city_title = xgettext('г. <title/>', title=settlement.L_title)

            if len(stations) == 1:
                station = stations[0]

                content.append({
                    'elem': 'line',
                    'mix': [{'block': 'i-pressed-controller'}],
                    'content': [{
                        'elem': 'item',
                        'content': [{
                            'block': 'b-link',
                            'url': langify(reverse('station', kwargs={'sid': station.id})),
                            'content': {
                                'elem': 'item',
                                'content': city_title
                            }
                        }]
                    }]
                })

                continue

            content_part = [{
                'elem': 'line',
                'mix': [{'block': 'i-pressed-controller'}],
                'content': [{
                    'elem': 'item',
                    'content': [{
                        'block': 'b-link',
                        'url': langify(reverse('city', kwargs={'cid': settlement.id})),
                        'content': {
                            'elem': 'item',
                            'content': city_title
                        }
                    }]
                }]
            }]

            for station in stations:
                content_part.append({
                    'elem': 'line',
                    'mix': [{'block': 'i-pressed-controller'}],
                    'content': [{
                        'elem': 'item',
                        'mods': {'stage': 'two'},
                        'content': [{
                            'block': 'b-link',
                            'url': langify(reverse('station', kwargs={'sid': station.id})),
                            'content': {
                                'elem': 'item',
                                'content': xgettext('а/п <station/>', station=station.L_title)
                            }
                        }]
                    }]
                })

            content.extend(content_part)

        return {
            'block': 'b-routers',
            'elemMods': {'type': 'search-result'},
            'content': content
        }

    def content(self):
        return [
            {
                'elem': 'item',
                'content': [
                    self.input()
                ]
            },
            {
                'elem': 'item',
                'content': self.precise_content()
            }
        ]
