# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from datetime import datetime
from django.core.urlresolvers import reverse

from common.models.transport import TransportType
from common.utils.avia import get_avia_thread_url

from common.xgettext.i18n import gettext, ngettext, xgettext
from common.utils.locations import composeurl, langify

from ..base import Base
from ..blocks import ErrorDisclaimerBlock, DetailsRouter, DetailsAboutLocalTime
from ..utils import wrap_with_link_item, punctuate_content
from .search import TimetableBlock


class BaseTemplate(Base):

    def e_search(self):
        return {
            'block': 'b-wrapper',
            'attrs': {'action': langify(reverse('search_threads')), 'method': 'get'},
            'mods': {'type': 'search-input'},
            'content': {
                'block': 'b-search',
                'content': [
                    {
                        'elem': 'input',
                        'content': {
                            'block': 'input',
                            'mods': {'theme': 'city', 'size': 'm'},
                            'content': [
                                {
                                    'elem': 'label',
                                    'attrs': {'for': 'input-search-trip'},
                                    'content': gettext('Вы искали')
                                },
                                {
                                    'elem': 'control',
                                    'attrs': {
                                        'id': 'input-search-trips',
                                        'name': 'number',
                                        'placeholder': gettext('Рейс или поезд №'),
                                        'value': self.context.query
                                    },
                                }
                            ]
                        }
                    },
                    {
                        'elem': 'button',
                        'content': {
                            'block': 'button',
                            'mods': {'theme': 'search', 'size': 'xl'},
                            'attrs': {'type': 'submit', 'role': 'submit'},
                            'content': gettext('Найти')
                        }
                    }
                ]
            }
        }

    def content(self):
        return [{
            'elem': 'item',
            'content': self.e_search()
        }, self.search_content()]

    def search_content(self):
        return []


class Template(BaseTemplate):
    page = 'search-result-on-name-trip'

    def link_content(self, code, n):
        text = None

        if code == 'bus':
            text = ngettext(n, '%d Автобус', '%d Автобуса', '%d Автобусов') % n

        if code == 'train':
            text = ngettext(n, '%d Поезд', '%d Поезда', '%d Поездов') % n

        if code == 'plane':
            text = ngettext(n, '%d Самолёт', '%d Самолёта', '%d Самолётов') % n

        if code in TransportType.WATER_TTYPE_CODES:
            text = ngettext(n, '%d Теплоход', '%d Теплохода', '%d Теплоходов') % n

        if code == 'suburban':
            text = ngettext(n, '%d Электричка', '%d Электрички', '%d Электричек') % n

        if code == 'helicopter':
            text = ngettext(n, '%d Вертолёт', '%d Вертолёта', '%d Вертолётов') % n

        return [
            {
                'block': 'b-icon-carrier',
                'mods': {'type': code}
            },
            text
        ]

    def search_content(self):
        query = self.context.query
        form = self.context.form
        groups = self.context.t_type_groups

        if not form.is_valid():
            return [ErrorDisclaimerBlock(form.list_errors())]

        if not groups:
            return [ErrorDisclaimerBlock(gettext('Такого рейса не существует.<br/>Попробуйте поискать другой рейс или маршрут.'))]

        return [{
            'block': 'b-transport',
            'mix': {'block': 'b-list-transport'},
            'content': [{
                'elem': 'line',
                'content': [{
                    'elem': 'item',
                    'elemMods': {'column': 'left'},
                    'content': [{
                        'block': 'b-link',
                        'url': composeurl('search_threads', params={'number': query, 'ttype': code}),
                        'mix': [{'block': 'b-transport', 'elem': 'title'}],
                        'content': self.link_content(code, cnt)
                    }]
                }]
            } for code, cnt in groups.iteritems()]
        }]

    def content(self):
        return [{
            'elem': 'item',
            'content': [self.e_search()] + self.search_content()
        }]


class OrdinalSearchTemplate(BaseTemplate):
    page = 'search-result-avia-on-name-trip'

    def line(self, thread, exact=False):
        t_code = self.context.t_code

        transporter_content = []

        if thread.L_title_special():
            transporter_content.extend([
                {
                    'elem': 'title-special',
                    'attrs': {'style': 'font-size: 12px;'},
                    'content': punctuate_content(
                        ['«', thread.L_title_special(), '»'],
                        ''
                    )
                },
                ', '
            ])

        if thread.company:
            transporter_content.extend([thread.company.L_title(), ', '])

        if getattr(thread, 'first_station', None):
            departure_time = thread.first_station.departure_time.strftime('%H:%M')
            schedule = thread.first_station.schedule

            transporter_content.append(departure_time)

            if getattr(thread, 'last_station', None):
                arrival_time = thread.last_station.arrival_time.strftime('%H:%M')

                transporter_content.extend([' - ', arrival_time])

            transporter_content.append(' ')

            transporter_content.extend(wrap_with_link_item(schedule))

        return {
            'elem': 'line',
            'elemMods': exact and {'first': 'yes'} or None,
            'js': True,
            'content': [
                {
                    'elem': 'item',
                    'content': [{
                        'block': 'b-link',
                        'url': get_thead_url(thread, self.request.tld),
                        'mix': [{'block': 'b-routers', 'elem': 'number-transport'}],
                        'content': [
                            {
                                'block': 'b-icon-carrier',
                                'elemMods': {'type': t_code}
                            },
                            thread.number
                        ]
                    }]
                },
                {
                    'elem': 'item',
                    'elemMods': {'type': 'route'},
                    'content': [
                        {
                            'block': 'b-link',
                            'mix': [{'block': 'b-route'}],
                            'content': wrap_with_link_item(thread.L_title())
                        },
                        {
                            'elem': 'transporter',
                            'content': transporter_content
                        }
                    ]
                }
            ]
        }

    def search_content(self):
        form = self.context.form
        threads = self.context.threads
        exact = self.context.exact
        content = []

        if not form.is_valid:
            return [ErrorDisclaimerBlock(form.list_errors())]

        if not threads and not exact:
            return [ErrorDisclaimerBlock(gettext('Такого рейса не существует.<br/>Попробуйте поискать другой рейс или маршрут.'))]

        if exact:
            content.extend([
                self.line(exact, True),
                {
                    'elem': 'line',
                    'elemMods': {'empty': 'yes'},
                    'content': [
                        {'elem': 'item'},
                        {'elem': 'item'},
                    ]
                }
            ])

        content += [self.line(thread) for thread in threads]

        return [
            {
                'elem': 'item',
                'content': DetailsAboutLocalTime()
            },
            {
                'block': 'b-routers',
                'mods': {'type': 'search-number-trip'},
                'content': content
            },
        ]


class SpecialSearchTemplate(BaseTemplate):
    page = 'search-result-train'
    show_only_aeroexpress = False

    def timetables(self):
        search_results = self.context.search_results
        content = []

        if not search_results:
            return []

        self.context.form = None

        for search_result in search_results:
            self.context.segments = search_result['segments']
            self.context.point_from = search_result['point_from']
            self.context.point_to = search_result['point_to']
            self.context.when = datetime.today()

            t_block = TimetableBlock(self,
                                     show_days=True,
                                     show_prices=False,
                                     show_tariffs_column=False,
                                     show_time_left=False,
                                     ajax=False,
                                     hide_gones=False,
                                     show_not_found_message=False,
                                     show_date_if_changed=False,
                                     mods={'search-threads': 'yes'})

            content += [
                {
                    'block': 'b-title',
                    'attrs': {'id': search_result['fragment']},
                    'mods': {'size': 'middle'},
                    'content': search_result['title']
                },
                t_block
            ]

        return content

    def links(self):
        links = []

        for row in self.context.search_navigation:
            if not row:
                continue

            links.extend([link for link in row if link])

        return {
            'block': 'arrow-box',
            'mods': {'type': 'native-select'},
            'attrs': {'style': 'margin-right: 10px;'},
            'arrowOpt': [{
                'name': ''.join(link['link_text']),
                'value': ''.join(link['link_text']),
                'link': '#' + link['fragment'],
                'current': not i
            } for i, link in enumerate(links)],
            'content': [
                '',
                {
                    'elem': 'selected',
                    'content': links[0]['link_text'],
                    'value': '#' + links[0]['fragment']
                }
            ]
        }

    def search_content(self):
        return [{
            'elem': 'item',
            'content': [
                {
                    'block': 'b-title',
                    'content': xgettext('Расписание поездов «<number/>»', number=self.context.number)
                },
                DetailsRouter(gettext('''Это расписание на все дни. Для получения информации о стоимости проезда
                                         и наличии мест укажите конкретную дату выезда.''')),
                self.links(),
                DetailsAboutLocalTime()
            ]},
            self.timetables()
        ]


def get_thead_url(thread, tld):
    if thread.t_type_id == TransportType.PLANE_ID:
        return get_avia_thread_url(thread, tld, utm_medium='main', utm_campaign='fligh_search')
    else:
        return composeurl('thread', kwargs={'uid': thread.uid})
