# -*- coding: utf-8 -*-

from __future__ import unicode_literals
from django.core.urlresolvers import reverse

from common.utils.locations import langify
from common.xgettext.i18n import gettext
from common.utils.date import uni_strftime

from travel.rasp.touch.touch.core.templates.blocks import ErrorDisclaimerBlock
from travel.rasp.touch.touch.core.templates.base import Base
from travel.rasp.touch.touch.core.templates.blocks.switcher import SwitcherTemplate
from ..utils import NBSP


class TimeWithDateIfChangedFormatter(object):

    def __init__(self):
        self.prev_date = None

    def __call__(self, dt, hidden=False):
        if dt.date() != self.prev_date:
            self.prev_date = dt.date()
            fmt = '%d' + NBSP + '%B %H:%M'

            return uni_strftime(fmt, dt)

        return uni_strftime('%H:%M', dt)


class StationBase(Base):
    def __init__(self, *args, **kwargs):
        self.time_formatter = TimeWithDateIfChangedFormatter()

        super(StationBase, self).__init__(*args, **kwargs)

    def schedule_tabs_switcher(self, active_type):
        if len(self.context.types) <= 1:
            return

        items = [{'js': {'value': ttype, 'url': url}, 'text': title} for (ttype, url, title) in self.context.types]

        return {
            'block': 'b-wrapper',
            'mods': {'type': 'radio-button', 'scoreboard': 'yes'},
            'content': SwitcherTemplate(value=active_type,
                                        items=items,
                                        mods={'size': 's', 'switch': 'types'})
        }

    def event_tabs_switcher(self):
        items = [
            {
                'js': {'value': 'departure'},
                'text': gettext('Отправление')
            },
            {
                'js': {'value': 'arrival'},
                'text': gettext('Прибытие')
            }
        ]

        return {
            'block': 'b-scoreboard',
            'content': [{
                'elem': 'switcher-place',
                'content': SwitcherTemplate(self.context.event, items,
                                            param_name='event')
            }]
        }

    def b_routers(self):
        return {
            'block': 'b-routers',
            'mods': {'type': 'for-scoreboard', 'show': 'fog'},
            'js': self.get_routes_js_params(),
            'content': [self.line(item, i) for i, item in enumerate(self.context.routes)]
        }

    def get_common_routes_js_params(self):
        station = self.context.station

        return {
            'ttype': station.t_type.code,
            'event': self.context.event,
            # максимальные даты по поиску с запросом
            'minDate': self.context.min_date,
            'maxDate': self.context.max_date,
            # текущее окно табло
            'startDate': self.context.start_date,
            'endDate': self.context.end_date,
            # параметры для обновления расписания
            'scheduleType': self.context.schedule_type,
            'update-url': langify(reverse('station_routes', kwargs={
                'sid': station.id,
                'schedule_type': self.context.schedule_type,
                'mode': self.context.mode
            }))
        }

    def get_routes_js_params(self):
        station = self.context.station

        return {
            'ttype': station.t_type.code,
            'event': self.context.event,
            # максимальные даты по поиску с запросом
            'minDate': self.context.min_date,
            'maxDate': self.context.max_date,
            # текущее окно табло
            'startDate': self.context.start_date,
            'endDate': self.context.end_date,
            # параметры для обновления расписания
            'scheduleType': self.context.schedule_type,
            'update-url': langify(reverse('station_routes', kwargs={
                'sid': station.id,
                'schedule_type': self.context.schedule_type,
                'mode': self.context.mode
            }))
        }


class NoScheduleErrorTemplate(Base):
    page = 'scoreboard-airport'

    def content(self):
        station = self.context.station

        return [
            {
                'elem': 'item',
                'elemMods': {'type': 'scoreboard'},
                'content': [
                    {
                        'block': 'b-transport-head',
                        'mods': {'type': 'scoreboard'},
                        'content': [
                            {
                                'elem': 'title',
                                'content': [
                                    {
                                        'elem': 'part-title',
                                        'content': gettext('Расписание')
                                    },
                                    {
                                        'elem': 'part-title',
                                        'content': {
                                            'block': 'b-link',
                                            'url': '/info/station/%s/' % station.id,
                                            'content': station.L_title()
                                        }
                                    }
                                ]
                            },
                        ]
                    },
                ]
            },
            ErrorDisclaimerBlock(
                gettext(u'''У Яндекс.Расписаний нет сведений о&nbsp;рейсах, останавливающихся на&nbsp;этой станции.''')
            )
        ]
