# -*- coding: utf-8 -*-

from __future__ import absolute_import
from __future__ import unicode_literals

from datetime import timedelta

from django.conf import settings
from common.models.transport import TransportType

from common.apps.suburban_events.api import EventStateType
from common.utils.date import FuzzyDateTime, NBSP, get_pytz, uni_strftime
from common.utils.locations import composeurl
from common.xgettext.i18n import gettext, ngettext, tgettext, xformat, xgettext

from travel.rasp.touch.modelBuilders.thread.thread_graphic import get_thread_graphic
from travel.rasp.touch.tariffs.utils import CLS_NAMES
from travel.rasp.touch.touch.core.templates.base import Base
from travel.rasp.touch.touch.core.templates.blocks.buy_train_ticket import BuyTrainTicket
from travel.rasp.touch.touch.core.templates.blocks.buy_avia_ticket import BuyAviaTicket, AviaDisclaimer
from travel.rasp.touch.touch.core.templates.blocks.companyIcon import CompanyIconModel, CompanyIconTemplate
from travel.rasp.touch.touch.core.templates.utils import get_station_title
from travel.rasp.touch.touch.utils import station_ttype_codes, get_station_url
from .utils import wrap_with_link_item, human_duration, currency
from .blocks import DetailsAboutLocalTime

DISABLE_AVIA_ORDER_INFO = True

class TimeWithDateIfChangedFormatter(object):

    def __init__(self):
        self.prev_date = None

    def __call__(self, dt, force_show_date=False):
        fmt = '%H:%M'

        if isinstance(dt, FuzzyDateTime):
            fmt = '~' + fmt

        if force_show_date:
            fmt += ' <span style="font-weight: 100;">%d' + NBSP + '%B</span>'

        elif dt.date() != self.prev_date:
            self.prev_date = dt.date()

            fmt += ' <span style="font-weight: 100;">%d' + NBSP + '%B</span>'

        return uni_strftime(fmt, dt)


class Template(Base):
    page = 'route-bus'
    share = True

    def __init__(self, *args, **kwargs):
        super(Template, self).__init__(*args, **kwargs)

        self.time_formatter = TimeWithDateIfChangedFormatter()

    def head_content_extended(self):
        uid = self.context.thread.uid

        if '_af' in uid:
            return [{'elem': 'meta', 'attrs': {'name': 'apple-itunes-app', 'content': 'app-id=387272416'}}]

        return []

    def b_station(self, rtstation, position, color, inputs_color, event, force_show_dt):
        thread = self.context.thread
        start_dt = self.context.start_dt
        station = rtstation.station
        ttype = thread.t_type.code
        city = station.settlement
        station_ttypes = station_ttype_codes(station)

        scheme_content = None
        if position == 'begin':
            scheme_content = [
                {'elem': 'point'},
                {'elem': 'road', 'elemMods': {'position': 'after'}}
            ]

        if position == 'middle':
            scheme_content = [
                {'elem': 'road', 'elemMods': {'position': 'before', 'color': inputs_color[0]}},
                {'elem': 'point'},
                {'elem': 'road', 'elemMods': {'position': 'after', 'color': inputs_color[1]}}
            ]

        if position == 'end':
            scheme_content = [
                {'elem': 'road', 'elemMods': {'position': 'before'}},
                {'elem': 'point'},
            ]

        loc_event_dt = rtstation.get_loc_event_dt(event, start_dt)

        if rtstation.is_fuzzy:
            if rtstation.tz_arrival and rtstation.tz_departure and event == 'arrival':
                loc_dep_dt = rtstation.get_loc_departure_dt(start_dt)

                stop_time = loc_dep_dt - loc_event_dt

                loc_dep_dt_fuzzy = FuzzyDateTime.fuzz(loc_dep_dt)

                loc_event_dt = FuzzyDateTime(loc_dep_dt_fuzzy - stop_time)

            else:
                loc_event_dt = FuzzyDateTime.fuzz(loc_event_dt)

        time = lambda: {'elem': 'time', 'content': self.time_formatter(loc_event_dt, force_show_dt)}

        if event == 'departure':
            if ttype == 'plane':
                arr_dep_text = xgettext('вылет <time/>', time=time)

            else:
                arr_dep_text = xgettext('отпр. <time/>', time=time)

        else:
            arr_dep_text = xgettext('приб. <time/>', time=time)

        title = get_station_title(station, national_version=self.context.request.NATIONAL_VERSION)
        if ttype == 'plane' and city:
            title = '%s (%s)' % (city.L_title(), title)

        if rtstation.is_technical_stop:
            title += ' <sup>(!)</sup>'

        if ttype in ('train', 'suburban') and ttype in station_ttypes:
            url = get_station_url(kwargs={'sid': station.id, 'type_code': ttype})

        else:
            url = get_station_url(kwargs={'sid': station.id})

        event_state_data = self.get_event_state_data(rtstation, event, loc_event_dt)

        return {
            'block': 'b-station',
            'mods': {'location': position},
            'content': [
                {
                    'elem': 'illustration',
                    'content': [{
                        'block': 'b-route-scheme',
                        'mods': {'color': color, 'position': position},
                        'content': scheme_content
                    }]
                },
                {
                    'elem': 'description',
                    'content': [
                        {
                            'elem': 'city',
                            'elemMods': ttype == 'plane' and {'color': 'blue'} or None,
                            'content': [{
                                'block': 'b-link',
                                'url': url,
                                'mods': rtstation.is_technical_stop and { 'asterisk': 'show' } or None,
                                'content': [
                                    {'elem': 'text', 'content': title}
                                ]
                            }]
                        },
                        {
                            'elem': 'arrival-departure',
                            'content': [
                                arr_dep_text
                            ]
                        },
                        event_state_data and {
                            'elem': 'arrival-departure',
                            'elemMods': {'type': 'ok' if event_state_data['type'] == 'ok' else 'delay'},
                            'content': [event_state_data['content']]
                        }
                    ]
                }
            ]
        }

    def b_station_other(self, color, stations):
        n = len(stations)
        js_models = [self.station_js(rt_station, color) for rt_station in stations]

        return {
            'block': 'b-station',
            'mods': {'location': 'middle'},
            'content': [
                {
                    'elem': 'illustration',
                    'content': [{
                        'block': 'b-route-scheme',
                        'mods': {'position': 'middle', 'with-other-stations': 'yes', 'color': color},
                        'content': [
                            {'elem': 'road', 'elemMods': {'position': 'before'}},
                            {'elem': 'other'},
                            {'elem': 'road', 'elemMods': {'position': 'after'}}
                        ]
                    }]
                },
                {
                    'elem': 'description',
                    'content': [{
                        'elem': 'more-stations',
                        'content': {
                            'block': 'b-action-button',
                            'mods': {'type': 'stations'},
                            'js': {
                                'stations': js_models
                            },
                            'content': ngettext(n, '%d остановка', '%d остановки', '%d остановок') % n
                        }
                    }]
                }
            ]
        }

    def station_js(self, rtstation, color):
        start_dt = self.context.start_dt

        arrival_loc_dt = rtstation.get_arrival_loc_dt(start_dt)
        departure_loc_dt = rtstation.get_departure_loc_dt(start_dt)

        if rtstation.is_fuzzy:
            arrival_loc_dt = FuzzyDateTime.fuzz(arrival_loc_dt)
            departure_loc_dt = FuzzyDateTime.fuzz(departure_loc_dt)

        arrival_time = self.time_formatter(arrival_loc_dt)
        departure_time = self.time_formatter(departure_loc_dt)

        stop_td = timedelta(0)

        if rtstation.tz_arrival is not None and rtstation.tz_departure is not None:
            stop_td = timedelta(minutes=rtstation.tz_departure - rtstation.tz_arrival)

        tcode = self.context.thread.t_type.code

        kwargs = {
            'sid': rtstation.station_id
        }

        if tcode in ('train', 'suburban'):
            kwargs['type_code'] = tcode

        url = get_station_url(kwargs=kwargs)

        stop_duration = stop_time_text = state_type = state_content = None

        has_stop = arrival_loc_dt != departure_loc_dt
        if has_stop:
            if tcode == 'train' or stop_td >= timedelta(minutes=2):
                stop_duration = human_duration(stop_td)
                stop_time_text = '{}-{}'.format(arrival_time, departure_time)
            else:
                stop_time_text = xgettext('отпр. <time/>', time={'elem': 'time', 'content': departure_time})

            state_type, state_content = self.get_rtstation_state(rtstation, arrival_loc_dt, departure_loc_dt)

        show_url = (not rtstation.station.hidden and
                    not rtstation.station.not_in_search() and
                    rtstation.station.interesting())

        title = get_station_title(rtstation.station, national_version=self.context.request.NATIONAL_VERSION)

        return {
            'id': rtstation.station_id,
            'title': title,
            'color': color,
            'url': url if show_url else None,
            'hasLongStop': stop_td.seconds > 540,  # выделяем времена стоянки больше 9 минут
            'hasStop': has_stop,
            'hasTechnicalStop': rtstation.is_technical_stop,
            'stateContent': state_content,
            'stateType': state_type,
            'stopDuration': stop_duration,
            'stopTimeText': stop_time_text,
        }

    def content(self):
        return [
            {
                'elem': 'item',
                'content': [
                    self.transport_head(),
                    DetailsAboutLocalTime(True),
                    self.details()
                ]
            },
            self.disclaimer()
        ]

    def disclaimer(self):
        return None

    def transport_head(self):
        company = self.context.company

        return {
            'block': 'b-transport-head',
            'mods': {'type': 'share'},
            'content': [
                {
                    'elem': 'button',
                    'content': {
                        'block': 'b-board-route',
                        'mods': {'type': 'in-head'},
                        'content': [{
                            'block': 'b-action-button',
                            'mods': {'type': 'share'}
                        }]
                    }
                },
                self.title_elem(),
                self.b_company(company)
            ]
        }

    def b_company(self, company):
        if company and not company.hidden:
            if company.strange:
                return [
                    company.L_title(),
                    CompanyIconTemplate(CompanyIconModel.get_icon_from_company(company))
                ]
            else:
                return {
                    'block': 'b-link',
                    'mods': {'carrier': 'yes'},
                    'url': composeurl('info', kwargs={'object_type': 'company', 'object_id': company.id}),
                    'content': [
                        company.L_title(),
                        CompanyIconTemplate(CompanyIconModel.get_icon_from_company(company))
                    ]
                }

    def title_elem(self):
        thread = self.context.thread
        ttype = thread.t_type.code

        if ttype in TransportType.WATER_TTYPE_CODES:
            ttype = 'water'

        titles = {
            'bus': gettext('Автобус'),
            'suburban': gettext('Пригородный поезд'),
            'train': gettext('Поезд'),
            'plane': gettext('Самолёт'),
            'water': gettext('Теплоход')
        }

        return {
            'elem': 'title',
            'elemMods': {'nowrap': 'no'},
            'content': [
                {
                    'elem': 'part-title',
                    'content': titles.get(ttype)
                },
                NBSP,
                ttype in ('train', 'suburban', 'plane') and {
                    'elem': 'part-title',
                    'elemMods': {'type': 'number-transport'},
                    'content': thread.number
                },
                NBSP,
                thread.L_title_special() and {
                    'elem': 'part-title',
                    'elemMods': {'type': 'name-transport'},
                    'content': '«%s»' % thread.L_title_special()
                },
                NBSP,
                ttype == 'plane' and thread.t_model and {
                    'elem': 'part-title',
                    'elemMods': {'type': 'name-transport'},
                    'content': thread.t_model.title
                },
                NBSP,
                thread.is_aeroexpress or thread.is_express and {
                    'elem': 'part-title',
                    'elemMods': {'type': 'mode-transport'},
                    'mix': [{'block': 'b-routers', 'elem': 'attention'}],
                    'content': thread.is_express and gettext('экспресс') or gettext('аэроэкспресс')
                }
            ]
        }

    def b_graphic(self):
        rts_stations = self.context.rtstations
        station_from = self.context.station_from
        station_to = self.context.station_to
        thread = self.context.thread

        t_code = thread.t_type.code

        if thread.is_aeroexpress:
            t_code = 'train'

        color = settings.TTYPE_COLOR_MAP.get(t_code)

        result = []
        for scheme in get_thread_graphic(rts_stations, station_from, station_to):
            if scheme['type'] == 'scheme':
                result.append(self.b_scheme(scheme, color))
            if scheme['type'] == 'transfer':
                result.append(self.b_transfer_scheme(scheme))
        return result

    def b_scheme(self, scheme, scheme_color):
        stations = []
        nodes = scheme['nodes']
        i_last_node = len(nodes) - 1
        for i, node in enumerate(nodes):
            node_color = scheme_color if node['include_in_thread'] else None

            if node['type'] == 'station':
                position = 'middle'
                if i == 0:
                    position = 'begin'
                if i == i_last_node:
                    position = 'end'

                inputs_color = ['grey', 'grey']

                if node['include_in_thread']:
                    if i != 0 and nodes[i-1]['include_in_thread']:
                        inputs_color[0] = node_color
                    if i != i_last_node and nodes[i+1]['include_in_thread']:
                        inputs_color[1] = node_color

                station = node['station']
                show_full_date = self.is_change_day(i, nodes)
                stations.append(self.b_station(rtstation=station,
                                               position=position,
                                               color=node_color,
                                               inputs_color=inputs_color,
                                               event=node['event'],
                                               force_show_dt=show_full_date))

            if node['type'] == 'collapsed-station' and node['stations']:
                stations.append(self.b_station_other(color=node_color,
                                                     stations=node['stations'],
                                                     ))

        return {
            'elem': 'line',
            'elemMods': {'type': 'scheme'},
            'content': stations
        }

    def is_change_day(self, index, nodes):
        if index == 0:
            return None
        if index == len(nodes):
            start_st_node = nodes[0]
            end_st_node = nodes[-1]
        else:
            start_st_node = nodes[index - 1]['type'] if nodes[index - 1]['type'] == 'station' else nodes[index - 2]
            end_st_node = nodes[index]

        start_dt = self.context.start_dt

        loc_departure_d = start_st_node['station'].get_loc_departure_dt(start_dt).date()
        loc_arrival_d = end_st_node['station'].get_loc_arrival_dt(start_dt).date()

        return loc_arrival_d != loc_departure_d

    def b_transfer_scheme(self, scheme):
        rtstation = scheme['node']['station']
        stop_td = timedelta(0)

        if rtstation.tz_arrival is not None and rtstation.tz_departure is not None:
            stop_td = timedelta(minutes=rtstation.tz_departure - rtstation.tz_arrival)

        return {
            'elem': 'line',
            'elemMods': {'type': 'scheme'},
            'content': {
                'elem': 'description',
                'elemMods': {
                    'type': 'change'
                },
                'content': 'Пересадка {0}'.format(human_duration(stop_td))
            }
        }

    def train_tariffs(self, tariffs_info):
        return {
            'elem': 'description',
            'elemMods': {'type': 'table'},
            'mix': [{
                'elem': 'thread-tariff'
            }],
            'content': [{
                'elem': 'row-table',
                'content': [
                    {
                        'elem': 'item-table',
                        'content': CLS_NAMES[place.code]
                    },
                    {
                        'elem': 'item-table',
                        'content': {
                            'elem': 'highlighted-text',
                            'content': currency(place.tariff, self.context.currency_info, place.link)
                        }
                    },
                    {
                        'elem': 'item-table',
                        'content': ngettext(place.seats, u'%d место', u'%d места', u'%d мест') % place.seats
                    }
                ]
            } for place in sorted(tariffs_info.places, key=lambda x: x.tariff)]
        }

    def plane_tariff(self, place, price):
        if DISABLE_AVIA_ORDER_INFO:
            return

        return {
            'elem': 'description',
            'elemMods': {'type': 'table', 'transport': 'avia'},
            'mix': [{
                'elem': 'thread-tariff'
            }],
            'content': {
                'elem': 'row-table',
                'content': [
                    tgettext('Более подробную информацию об этом рейсе вы можете найти на сервисе'),
                    {
                        'block': 'b-link',
                        'mix': {
                            'block': 'b-details-router',
                            'elem': 'avia-link'
                        },
                        'url': place.deep_link,
                        'target': '_blank',
                        'content': tgettext('Яндекс.Авиабилеты')
                    }
                ]
            }
        }

    def tariff(self):
        thread = self.context.thread
        tcode = thread.t_type.code
        display_info = self.context.display_info
        if not display_info:
            return

        tariff_info = display_info.get('tariffs_info')

        if not tariff_info:
            return

        # !!! европейские поезда не имеют мест и классов
        if tcode == 'train' and all(p.code for p in tariff_info.places):
            return self.train_tariffs(tariff_info)

        if not tariff_info.places:
            return

        place = min(tariff_info.places, key=lambda x: x.tariff)

        price = currency(place.tariff, self.context.currency_info, getattr(place, 'min', False))

        if tcode == 'plane':
            return self.plane_tariff(place, price)

        return {
            'elem': 'description',
            'mix': [{
                'elem': 'thread-tariff'
            }],
            'content': [xgettext('Стоимость проезда <tariff/>',
                                 tariff=lambda: {'elem': 'highlighted-text', 'content': price})]
        }

    def details(self):
        thread_stops = self.context.thread_stops
        ttype = self.context.thread.t_type.code
        has_tech_stations = any(rts.is_technical_stop for rts in self.context.rtstations)
        mods = {}

        if ttype == 'train' and self.context.buy_context:
            mods['hidden-tariffs'] = 'yes'

        return {
            'block': 'b-details-router',
            'mods': mods,
            'content': self.b_graphic() + [

                {
                    'elem': 'line',
                    'content': [
                        {
                            'elem': 'description',
                            'elemMods': {'type': 'time'},
                            'content': xgettext('Время в пути: <travel-time/>',
                                                travel_time=human_duration(self.context.duration))
                        }
                    ]
                },
                ttype != 'plane' and {
                    'elem': 'line',
                    'elemMods': {'type': 'with-border'},
                    'content': [
                        {
                            'elem': 'description',
                            'elemMods': {'type': 'schedule-title'},
                            'content': gettext('Данный вариант расписания действует')
                        },
                        {
                            'elem': 'description',
                            'elemMods': {'type': 'schedule-text'},
                            'content': self.context.schedule_days
                        },
                        thread_stops and {
                            'elem': 'description',
                            'elemMods': {'type': 'schedule-title'},
                            'attrs': {'style': 'margin-top: 15px;'},
                            'content': gettext('Остановки')
                        },
                        thread_stops and {
                            'elem': 'description',
                            'content': wrap_with_link_item(thread_stops)
                        },
                        has_tech_stations and {
                            'elem': 'description',
                            'elemMods': {'type': 'asterisk'},
                            'attrs': {'style': 'margin-top: 15px;'},
                            'content': gettext('<sup>(!)</sup> &mdash; техническая остановка, купить билет до этой станции невозможно.')
                        },
                    ]
                },
                {
                    'elem': 'line',
                    'content': self.tariff()
                },
            ]
        }

    def get_event_state_data(self, rtstation, event, schedule_dt):
        states_by_rtstation = self.context.states_by_rtstation
        event_state = getattr(states_by_rtstation.get(rtstation), event, None)

        if event_state is None:
            return

        if event_state.type == EventStateType.FACT:
            fact_dt = get_pytz(event_state.tz).localize(event_state.dt).astimezone(rtstation.station.pytz)
            delay_duration = fact_dt - schedule_dt

            if delay_duration < timedelta(minutes=1):
                return {
                    'type': 'ok',
                    'dt': fact_dt,
                    'content': xgettext('по расписанию')
                }

            time_text = fact_dt.strftime('%H:%M')
            return {
                'type': 'fact_delay',
                'dt': fact_dt,
                'content': [
                    {'elem': 'wrapper-text', 'content': [xgettext('отправился в <time/>', time=time_text)
                                                         if event == 'departure' else
                                                         xgettext('прибыл в <time/>', time=time_text)]},
                    {'elem': 'wrapper-text', 'content': ['&nbsp;&middot;&nbsp;',
                                                         xgettext('опоздание на <delay/>',
                                                                  delay=human_duration(delay_duration))]}
                ]
            }

        if event_state.type == EventStateType.POSSIBLE_DELAY:
            return {
                'type': 'possible_delay',
                'content': xgettext('возможно опоздание')
            }

    def get_rtstation_state(self, rtstation, arrival_loc_dt, departure_loc_dt):
        arrival_state_data = self.get_event_state_data(rtstation, 'arrival', arrival_loc_dt)
        departure_state_data = self.get_event_state_data(rtstation, 'departure', departure_loc_dt)

        if departure_state_data:
            if (departure_state_data['type'] == 'fact_delay' and
                    arrival_state_data and arrival_state_data['type'] in ('ok', 'fact_delay') and
                    (departure_state_data['dt'] - arrival_state_data['dt']) >= timedelta(minutes=2)
                ):
                # показываем диапазон времён при наличии задержки на долгой остановке
                return departure_state_data['type'], [{
                    'elem': 'wrapper-text',
                    'content': ['{:%H:%M}-{:%H:%M}'.format(arrival_state_data['dt'], departure_state_data['dt'])]
                }] + departure_state_data['content'][1:]

            return departure_state_data['type'], departure_state_data['content']

        if arrival_state_data:
            return arrival_state_data['type'], arrival_state_data['content']

        return None, None


class ThreadWithTrainBuy(Template):
    page = 'route-train-buy'
    page_type = 'buy'
    share = True

    def fixed_content(self):
        return BuyTrainTicket(self.context.buy_context)


class ThreadWithAviaBuy(Template):
    page = 'route-avia-buy'
    page_type = 'avia-buy'
    share = True

    def disclaimer(self):
        context = self.context
        return AviaDisclaimer(
            context.request.NATIONAL_VERSION,
            hidden=context.buy_context['state'] != 'form',
            keyset_type='thread'
        )

    def fixed_content(self):
        return BuyAviaTicket(self.context.buy_context)
