# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import re
from datetime import timedelta

from common.xgettext.i18n import xgettext, tgettext
from common.models.currency import Price
from common.utils.date import NBSP

from travel.rasp.touch.touch.core.templates.utils.wrappers import *
from travel.rasp.touch.touch.core.templates.utils.desktop_links import *

WEEKDAY_NAMES = {
    'tr': [
        u'Pazartesi', u'Salı', u'Çarşamba', u'Perşembe', u'Cuma', u'Cumartesi', u'Pazar'
    ],
    'en': [
        'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday'
    ],
}

MONTH_NAMES = {
    'tr': [
        u'Ocak', u'Şubat', u'Mart', u'Nisan', u'Mayıs', u'Haziran', u'Temmuz', u'Ağustos', u'Eylül', u'Ekim', u'Kasım', u'Aralık'
    ],
    'en': [
        'January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December'
    ]
}


def human_duration(duration):
    if not duration or duration < timedelta():
        return None

    minutes = (duration.seconds + 1) / 60
    hours = 0
    days = duration.days

    if minutes > 59:
        hours = minutes / 60
        minutes = minutes % 60

    # Если есть дни, минуты округляем до часов
    if days:
        hours += (minutes + 60 - 1) / 60
        minutes = 0

        # Если получилось 24 часа, то прибавляем день и убираем часы
        if hours == 24:
            days += 1
            hours = 0

    blocks = []

    if days:
        # Продолжительность действия: 1 день. Может являться частью фразы, например, 1 день 2 часа
        blocks.append(tgettext(u'<days /> д.', days=unicode(days)))

    if hours:
        # Продолжительность действия: 1 час. Может являться частью фразы, например, 1 день 2 часа
        blocks.append(tgettext(u'<hours /> ч.', hours=unicode(hours)))

    if minutes and len(blocks) < 2:
        # Продолжительность действия: 1 минута. Может являться частью фразы, например, 2 часа 3 минуты
        blocks.append(tgettext(u'<minutes /> мин.', minutes=unicode(minutes)))

    return u' '.join(b.replace(' ', NBSP) for b in blocks)


def currency(price, currency_info, _from=False):
    UNIT_RE = re.compile(r'<unit>(.*?)</unit>', re.U)
    JS_REPLACE_RE = re.compile(r'\$(\d\d?)')

    # потому что логическое приведение цены через not price, для маленьких белорусских цен вида 0.48 BYN
    # будет факапить
    if price <= 0:
        return

    if not isinstance(price, Price):
        price = Price(price)

    price.rebase(currency_info.rates)

    selected = hasattr(currency_info, 'selected') and currency_info.selected or None

    if price.base_value is None or not selected:
        currency = price.currency
        value = price.value

    else:
        currency = selected
        value = price.base_value / currency_info.rates[selected]

    formatter = currency_info.formats[currency]
    if currency_info.country_base == 'TRY':
        formatter = TurkishCurrencyFormatter(formatter)

    result = formatter.format_value(value, show_cents=False)

    if _from:
        result = u''.join(xgettext(u'от <currency-result/>', currency_result=result))

    result = UNIT_RE.sub(JS_REPLACE_RE.sub(r'\\\1', u'$1'), result)

    return result


class TurkishCurrencyFormatter:
    """Для Турции нужно всегда отбрасывать центы"""

    def __init__(self, default_format):
        self.default_format = default_format

    def format_value(self, *args, **kwargs):
        try:
            return self._format_value(*args, **kwargs)
        except:
            return 'n/a'

    def _format_value(self, value, show_cents=True):
        if show_cents:
            whole, cents = divmod(int(round(value * 100)), 100)
        else:
            whole = int(round(value))
            cents = 0

        if whole > 0:
            whole_str = str(whole)

            whole_len = len(whole_str)

            whole = u" ".join(whole_str[max(i, 0):i + 3] for i in range(-(-whole_len % 3), whole_len, 3))

        if show_cents and cents:
            if whole:
                return self.default_format.L_template().replace("%d", "%s", 1) % (whole, cents)

            return self.default_format.L_template_cents() % cents

        return self.default_format.L_template_whole().replace("%d", "%s", 1) % whole


def human_list_ru(l, or_=True):
    if len(l) < 2:
        return l

    return punctuate_content(l[:-1], sep=u', ') + [u' или ' if or_ else u' и' + NBSP, l[-1]]


def human_list_tr(l):
    if len(l) < 2:
        return l

    return punctuate_content(l[:-1], sep=u', ') + [u' ve ', l[-1]]


def cut_direction_full_title(direction):
    regexp = re.compile(ur'направление|напрямок')

    return regexp.sub(u'напр.', direction.L_full_title())


def get_station_title(station, national_version=None):
    if national_version != 'tr':
        return station.L_title()

    return station.L_title_with_type_tr()


def wrap_span(value):
    return {
        'elem': 'span',
        'tag': 'span',
        'content': value
    }


def wrap_link_item(value):
    return {
        'block': 'b-link',
        'elem': 'item',
        'content': value
    }


def wrap_time_in_path(value):
    return {
        'elem': 'span',
        'tag': 'span',
        'mix': [
            {
                'block': 'b-routers',
                'elem': 'time',
                'elemMods': {'type': 'in-path'}
            }
        ],
        'content': value
    }


def wrap_with(value, func):
    if not value:
        return ''

    parts = value.replace(u' - ', u' — ').split(u' ')

    return punctuate_content(map(func, parts), sep=u' ')


def wrap_with_time_in_path(value):
    return wrap_with(value, wrap_time_in_path)


def wrap_with_span_item(value):
    return wrap_with(value, wrap_span)


def wrap_with_link_item(value):
    return wrap_with(value, wrap_link_item)


def punctuate_content(content, sep=', '):
    punctuated_content = []

    iterator = iter(content)

    try:
        item = iterator.next()

    except StopIteration:
        return []

    while True:
        punctuated_content.append(item)

        try:
            item = iterator.next()

        except StopIteration:
            break

        punctuated_content.append(sep)

    return punctuated_content
