# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from urlparse import SplitResult

from django.http import QueryDict
from django.conf import settings

from common.utils.locations import langify
from common.utils.ya import get_my_url
from url_mapper import TouchMapper, UrlMappingError


FULL_VERSION_PRIORITY_ID = 44  # http://wiki.yandex-team.ru/MyCookie/NomerBloka


def build_url(path, params=None):
    if params:
        query = params.urlencode()
    else:
        query = ''

    return langify(SplitResult('http', settings.RASP_DOMAIN, path, query, '').geturl())


def search_threads_url(request):
    return build_url('threads', request.GET.copy())


def search_url(request, search_type=None):
    path = 'search'
    params = request.GET.copy()

    if search_type:
        path += '/' + search_type

    if 'all_days' in request.GET:
        params['when'] = 'на все дни'

    return build_url(path, params)


def city_url(city=None, direction=False):
    if not city:
        return build_url('')

    path = 'city/%s' % city

    if direction:
        path += '/direction'

    return build_url(path)


def station_search_url(request):
    params = request.GET.copy()

    params['cityFrom'] = params.get('pointName', '')
    params['cityFromId'] = ''

    if 'pointName' in params:
        del params['pointName']

    if 'pointId' in params:
        del params['pointId']

    return build_url('station_search', params)


def station_suburban_url(request):
    path = request.path.replace('/suburban', '')
    params = QueryDict(None, mutable=True)
    params['type'] = 'suburban'
    params['span'] = 'day'

    if 'filter' in request.GET:
        fv = request.GET['filter']

        if fv != 'today':
            params['span'] = None

    if 'direction' in request.GET:
        direction = request.GET['direction']

        if direction == 'на все направления':
            params['direction'] = 'all'

        else:
            params['direction'] = direction

    return build_url(path, params)


def station_train_url(request):
    path = request.path.replace('/train', '')
    params = QueryDict(None, mutable=True)
    params['type'] = 'train'

    if 'span' in request.GET:
        fv = request.GET['span']

        if fv == 'tomorrow':
            params['span'] = 'tomorrow'

        elif fv == 'schedule':
            params['span'] = 'schedule'

    if 'event' in request.GET:
        params['event'] = request.GET['event']

    return build_url(path, params)


def station_url(request):
    if 'suburban' in request.path:
        return station_suburban_url(request)

    if 'train' in request.path:
        return station_train_url(request)

    params = QueryDict(None, mutable=True)
    params['span'] = 'day'

    if 'event' in request.GET:
        params['event'] = request.GET['event']

    if request.GET.get('filter') == 'all':
        params['span'] = 'schedule'

    return build_url(request.path, params)


def desktop_domain_link(request):
    path = request.path

    if path.startswith('/search-threads'):
        return search_threads_url(request)

    if path.startswith('/direction'):
        if 'fromName' in request.GET or 'fromId' in request.GET:
            return search_url(request, 'suburban')

        return city_url(request.client_city.id, True)

    if path.startswith('/suburban-directions') or path.startswith('/stations'):
        return city_url(request.GET.get('city'))

    if path.startswith('/station-search'):
        return station_search_url(request)

    if path.startswith('/station'):
        return station_url(request)

    return build_url(path, request.GET)


def desktop_alternate_link(request):
    rasp_domain = settings.NATIONAL_RASP_DOMAINS[request.tld]
    mapper = TouchMapper(rasp_domain, 'https')
    touch_url = settings.TOUCH_URL + request.get_full_path()
    try:
        return mapper.get_rasp_url(touch_url)
    except UrlMappingError:
        return None


def desktop_version_link(request):
    rasp_domain = settings.NATIONAL_RASP_DOMAINS[request.tld]
    alt_link = desktop_alternate_link(request)
    if alt_link is None:
        return 'https://' + rasp_domain
    return alt_link


def tune_desktop_alternate_link(request):
    main_link = desktop_alternate_link(request)
    if main_link:
        return get_my_url(request, block=FULL_VERSION_PRIORITY_ID, param=1, retpath=main_link)
    return None
