# -*- coding: utf-8 -*-

from itertools import chain

from django.shortcuts import get_object_or_404
from django.conf import settings
from django.utils.translation import get_language
from django.views.generic import View

from common.models.geo import Settlement, Station
from common.models.transport import TransportType
from common.xgettext.i18n import gettext, xgettext
from travel.rasp.touch.forms.city import CityForm

from travel.rasp.touch.forms.search import SearchForm
from travel.rasp.touch.touch.core.helpers.transport_type import get_plural_transport_by, get_plural_station_name_by
from travel.rasp.touch.touch.core.templates import stations as stations_tpl, city as city_tpl
from travel.rasp.touch.touch.core.templates.utils import punctuate_content, human_list_tr, human_list_ru

from travel.rasp.library.python.common_old.settlement import get_stations_by_type


class CityView(View):

    def get_human_list(self, local_city_t_titles):
        lang = get_language()

        if lang == 'tr':
            return human_list_tr(local_city_t_titles)

        if lang == 'ua':
            return punctuate_content(local_city_t_titles)

        return human_list_ru(local_city_t_titles, or_=False)

    def generate_context(self, request, cid=None):
        if cid:
            city = get_object_or_404(Settlement.hidden_manager, pk=cid)

        else:
            city = request.client_city

        stations = get_stations_by_type(city, use_water=True)
        station = None
        all_stations = [station
                        for by_type in stations.values()
                        for station, terminals in chain(by_type['stations'], by_type['related'])]

        big = len(all_stations) > 1 or city.suburban_zone_id or city.region_id == Settlement.get_default_city().region_id

        if all_stations:
            if not big and all_stations[0].tablo_state:
                station = all_stations[0]

        if not stations['plane']['stations']:
            stations['plane']['stations'] = stations['plane']['related']

        aeroexpresses = [(st, terminals) for st, terminals in stations['plane']['stations'] if st.has_aeroexpress]

        if city.country_id not in settings.OUR_COUNTRIES:
            stations = {
                'plane': stations['plane']
            }

        t_type_titles = [
            ('plane', gettext(u'самолётов')),
            ('train', gettext(u'поездов')),
            ('bus', gettext(u'автобусов')),
            ('water', gettext(u'теплоходов'))
        ]

        city_t_titles = []

        for code, title in t_type_titles:
            if title in city_t_titles:
                continue

            if code in stations and stations[code]['stations']:
                city_t_titles.append(title)

        suburban_directions = self.get_suburban_direction(city, national_version=request.NATIONAL_VERSION)
        if suburban_directions:
            city_t_titles.append(gettext(u'электричек'))

        city_t_titles_func = lambda **kwargs: ''

        if city_t_titles:

            def city_t_titles_func_list(**kwargs):
                local_city_t_titles = city_t_titles[:]

                return self.get_human_list(local_city_t_titles)

            city_t_titles_func = city_t_titles_func_list

        elif station:
            city_t_titles_func = lambda **kwargs: get_plural_station_name_by(station.t_type.code)

        return {
            'stations': stations,
            'form': SearchForm(request=request),
            'city': city,
            'directions': suburban_directions,
            'aeroexpresses': aeroexpresses,
            'page_title': xgettext(u'<city-title/>. Расписание <ttypes-plural case="genitive"/>',
                                   city_title=city.L_title,
                                   ttypes_plural=city_t_titles_func),
            'is_main': True
        }

    def get_suburban_direction(self, city, national_version=None):
        if national_version == 'tr':
            return []

        if city.suburban_zone:
            directions = list(city.suburban_zone.externaldirection_set.all().order_by('title'))

            if directions:
                directions = [(u'', gettext(u'Не знаю направление'))] + directions
                return directions

        return []

    def get(self, request, cid=None):
        context = self.generate_context(request, cid)

        return city_tpl.City.render(request, context)


def get_stations(settlement, tcode, aeroex=False):
    t_types = {tcode}
    if tcode in TransportType.WATER_TTYPE_CODES:
        t_types = set(TransportType.WATER_TTYPE_CODES)

    qs = Station.objects.filter(hidden=False, majority__id__lt=3, t_type__code__in=t_types)

    if aeroex:
        qs = qs.filter(has_aeroexpress=True)

    stations = list(set(chain(qs.filter(settlement=settlement),
                              qs.filter(station2settlement__settlement=settlement))))

    def key_func(s):
        return s.majority_id, s.get_clean_title().strip('"'), s.id

    if tcode == 'plane' and not stations:
        stations = [rel.station for rel in settlement.related_stations.filter(station__hidden=False,
                                                                              station__majority__id__lt=3,
                                                                              station__t_type__code__in=t_types)]

    stations.sort(key=key_func)

    return stations


def stations(request, ttype):
    if ttype in TransportType.WATER_TTYPE_CODES:
        ttype = 'water'

    aeroex = 'aeroex' in request.GET

    city = CityForm(request).city

    stations = get_stations(city, ttype, aeroex)

    extension_ttype = ttype
    if aeroex and ttype == 'plane':
        extension_ttype = 'aeroex'

    title = get_plural_transport_by(extension_ttype)

    context = {
        'ttype': ttype,
        'city': city,
        'stations': stations,
        'aeroex': aeroex,
        'title': title,
        'page_title': title
    }

    return stations_tpl.Template.render(request, context)


def directions(request):
    city = CityForm(request).city

    main_directions, directions = city.get_directions()

    all_directions = list(chain(main_directions, directions))

    context = {
        'directions': all_directions,
        'city': city,
        'page_title': gettext(u'Направления'),
        'title': gettext(u'Направления')
    }

    return stations_tpl.DirectionsTemplate.render(request, context)
