# -*- coding: utf-8 -*-

from common.models.schedule import TrainSchedulePlan
from travel.rasp.library.python.common23.date import environment
from common.utils.blablacar_utils import BLABLACAR_ALL_DAYS_DATE
from common.xgettext.i18n import gettext
from common.models.geo import ExternalDirection, ExternalDirectionMarker
from common.models.transport import TransportType
from common.utils.httpresponses import jsonp_response
from common.views.currency import fetch_currency_info
from common.views.tariffs import DisplayInfo
from travel.rasp.touch.forms.city import CityForm
import settings
from travel.rasp.touch.tariffs.views import fill_tariffs_info

from travel.rasp.touch.forms.search import DirectionSearchForm
from travel.rasp.touch.touch.core.helpers.blablacar import check_blablacar_allowed
from travel.rasp.touch.touch.core.templates import error as err_tpl, precise as precise_tpl, direction as direction_tpl
from travel.rasp.touch.touch.core.templates.utils import cut_direction_full_title
from .search.search import search_segments
from .teasers import TeaserSetTouch
from .utils import fetch_end_station_ids, format_page_title


def station_title(station, directions=None):
    title = u''

    if station.t_type_id == TransportType.PLANE_ID:
        title = u"а/п "

    title += station.get_popular_title()

    if directions and station.id in directions:
        title += u" (%s)" % cut_direction_full_title(directions[station.id])

    if station.near_metro:
        title += u' [м.%s]' % station.near_metro

    return title


def get_direction_stations(direction):
    markers = list(ExternalDirectionMarker.objects.filter(external_direction__code=direction,
                                                          station__hidden=False).\
                                                    select_related('station').\
                                                    order_by())
    branches_order = {}
    i = 1

    for marker in markers:
        if not marker.branch in branches_order:
            branches_order[marker.branch] = i
            i += 1

    markers.sort(key=lambda marker: (marker.branch and branches_order[marker.branch] or 0, marker.order) )
    last_branch = None
    stations_list = []

    for marker in markers:
        if marker.branch != last_branch:
            stations_list.append({
                'point_key': '',
                'title': '---'
            })
            last_branch = marker.branch

        stations_list.append({
            'point_key': marker.station.point_key,
            'title': station_title(marker.station)
        })

    return stations_list


@jsonp_response
def ajax_direction_stations(request, direction):
    direction = ExternalDirection.objects.select_related('suburban_zone').get(code=direction)

    base_station_title = station_title(direction.base_station)
    base_station_key = direction.base_station.point_key

    response = {
        'base_station': {
            'title': base_station_title,
            'key': base_station_key
        },
        'stations': get_direction_stations(direction.code)
    }

    return response


def get_directions_info(city, direction_code=None):
    info = {
        'direction': None,
        'directions': None,
        'stations_list': None
    }

    if direction_code:
        try:
            direction = ExternalDirection.objects.select_related('suburban_zone').get(code=direction_code)

            info['direction'] = direction

            info['base_station_key'] = direction.base_station.point_key
            info['base_station_title'] = station_title(direction.base_station)

            stations_list = get_direction_stations(direction_code)

            info['stations_list'] = stations_list

            sub_zone = direction.suburban_zone

            if sub_zone:
                directions = [(d.code, cut_direction_full_title(d))
                              for d in sub_zone.externaldirection_set.all().order_by('title')]

                if directions:
                    info['directions'] = [(None, gettext(u'Все направления'))] + directions

        except ExternalDirection.DoesNotExist:
            pass

    if not info['directions'] and city.suburban_zone:
        directions = [(d.code, cut_direction_full_title(d))
                      for d in city.suburban_zone.externaldirection_set.all().order_by('title')]

        if directions:
            info['directions'] = [(None, gettext(u'Все направления'))] + directions

    if not info['directions']:
        info['directions'] = [(None, gettext(u'Все направления'))]

    return info


def search(request):
    city = CityForm(request, enable_foreign_logic=True).city

    context = {
        'city': city
    }

    direction_code = request.GET.get('direction')

    context.update(get_directions_info(city, direction_code))

    form = DirectionSearchForm(request.GET.copy(), request=request)

    context['form'] = form

    if not form.is_valid():
        context.update({
            'page_title': gettext(u'Некорректные параметры поиска'),
            'error': form.list_errors()
        })

        return err_tpl.DirectionTemplate.render(request, context)

    form_data = form.cleaned_data

    if form_data['ambiguous']:
        context['type'] = 'direction_search'
        context['page_title'] = gettext(u'Уточните пункты отправления и прибытия')

        if form_data['from'].has_variants():
            context['field'] = 'from'
            context['variants'] = form_data['from'].variants

        else:
            context['field'] = 'to'
            context['variants'] = form_data['to'].variants

        return precise_tpl.Template.render(request, context)

    point_from = form_data['from'].point
    point_to = form_data['to'].point

    if form_data['all_days']:
        dep_date = None

    else:
        dep_date = form['when'].data

    context['page_title'] = format_page_title(point_from, point_to, dep_date, 'suburban')

    segments, service_types, extended_points = search_segments(point_from, point_to,
                                                               departure_date=dep_date,
                                                               transport_types=('suburban',))

    if form_data['all_days']:
        set_train_schedule_plan(segments)

    if form.cleaned_data['mode'] == 'nearest':
        loc_date = point_from.get_local_datetime(request.now)
        segments = [s for s in segments if s.departure >= loc_date][:settings.SUBURBAN_AMOUNT_IN_SEARCH]

    context['end_station_ids'] = fetch_end_station_ids([s.thread for s in segments if hasattr(s, 'thread')])

    if extended_points:
        context['extended'] = True
        context['requested_from'] = point_from
        context['requested_to'] = point_to

        point_from = extended_points[0]
        point_to = extended_points[1]

    context['point_from'] = point_from
    context['point_to'] = point_to

    currency_info = fetch_currency_info(request)

    context['currency_info'] = currency_info

    for segment in segments:
        segment.display_info = DisplayInfo()

    allow_blablacar = check_blablacar_allowed('suburban', point_from, point_to)
    supplements = ['blablacar']

    if dep_date:
        context['ajax_tariffs_info'] = fill_tariffs_info(segments, currency_info, supplements, point_from,
                                                         point_to, dep_date, allow_blablacar, request)
    else:
        blablacar_segments = []
        context['ajax_tariffs_info'] = fill_tariffs_info(blablacar_segments, currency_info, supplements, point_from,
                                                         point_to, BLABLACAR_ALL_DAYS_DATE, allow_blablacar, request)
        segments.extend(blablacar_segments)

    def sort_key(s):
        if not dep_date:
            return s.departure.time()

        return s.departure

    segments.sort(key=sort_key)

    context['segments'] = segments

    data = {
        'points': [point_from, point_to],
        'routes': segments
    }

    teasers = TeaserSetTouch(request, 'search_suburban', data)

    context['teasers'] = teasers

    return direction_tpl.SearchTemplate.render(request, context)


def main(request):
    city = CityForm(request, enable_foreign_logic=True).city

    context = {
        'city': city,
        'page_title': gettext(u'Электрички'),
        'form': DirectionSearchForm(request=request)
    }

    direction_code = request.GET.get('direction')

    context.update(get_directions_info(city, direction_code))

    if context['direction']:
        teasers = TeaserSetTouch(request, 'direction', context['direction'])

        context['teasers'] = teasers

    return direction_tpl.Template.render(request, context)


def set_train_schedule_plan(segments):
    current_plan, next_plan = TrainSchedulePlan.add_to_threads(
        [s.thread for s in segments if hasattr(s, 'thread')],
        environment.today()
    )

    for segment in segments:
        segment.next_plan = next_plan
