# -*- coding: utf-8 -*-

import json
import itertools

from datetime import datetime

from django.http import HttpResponseBadRequest
from django.conf import settings
from django.views.generic import View

from common.data_api.ticket_daemon.serialization.segment import Segment
from common.models.geo import Country, Station
from common.models.transport import TransportType
from common.utils import marketstat
from common.utils.httpresponses import jsonp_response
from common.views.settlement import get_biggest_cities
from common.views.currency import fetch_currency_info
from common.views.tariffs import DisplayInfo
from common.xgettext.i18n import gettext, xgettext, dynamic_gettext
from geosearch.views.pointlist import PointList
from geosearch.views.pointtopoint import process_points_lists
from route_search.shortcuts import search_routes
from travel.rasp.touch.tariffs.views import fill_tariffs_info, search_tariffs
from travel.rasp.touch.tariffs.retrieving.base import DirectionAjaxInfo, ExtraTrainSegment

from travel.rasp.touch.forms.search import SearchForm
from travel.rasp.touch.touch.core.helpers.blablacar import check_blablacar_allowed
from travel.rasp.touch.touch.core.templates.search import search as search_tpl
from travel.rasp.touch.touch.core.templates.search import untypedSearch as untypedSearch_tpl
from travel.rasp.touch.touch.core.templates.search.untypedSearch import UntypedTimetableBlock
from travel.rasp.touch.touch.core.templates.utils import currency
from travel.rasp.touch.touch.core.templates import error as err_tpl, precise as precise_tpl
from travel.rasp.touch.touch.core.templates.parts import plane_tariff_places, tariffs_places
from ..teasers import TeaserSetTouch
from ..utils import fetch_end_station_ids, format_page_title, get_transport_code_from

search_logger = marketstat.Log(settings.SEARCH_LOG)
rasp_users_search_logger = marketstat.DsvLog(settings.RASP_USERS_SEARCH_LOG)


@jsonp_response
def ajax_tariffs(request, search_type):
    """обновление типизированной выдачи"""

    if request.method == 'POST':
        try:
            json_data = json.loads(request.body)

            direction_key = json_data['d']
            request_timestamp = json_data['ts']
            keys = json_data['k']
            ready_keys = json_data['r']
            ttype = json_data['ttype']
            borders = None

            point_from, point_to = DirectionAjaxInfo.decode_direction_key(direction_key)

            if 'b' in json_data:
                borders = [point_from.localize(loc=datetime.strptime(b, "%Y-%m-%d %H:%M")) for b in json_data['b']]

            if len(borders) != 2:
                borders = None

        except:
            return HttpResponseBadRequest('Bad request')

    else:
        try:
            direction_key = request.GET.get('d')

            point_from, point_to = DirectionAjaxInfo.decode_direction_key(direction_key)

            request_timestamp = json.loads(request.GET['ts'])

            keys = request.GET.getlist('k')

            try:
                ready_keys = json.loads(request.GET['r'])

            except KeyError:
                ready_keys = None

            try:
                bb = request.GET.getlist('b')

                borders = [point_from.localize(loc=datetime.strptime(b, "%Y-%m-%d %H:%M")) for b in bb]

                if len(borders) != 2:
                    borders = None

            except ValueError:
                borders = None

        except:
            return HttpResponseBadRequest('Bad request')

        ttype = request.GET.get('ttype')

    currency_info = fetch_currency_info(request)

    data, updates, extra_segments = search_tariffs(request, currency_info, point_from,
                                                   point_to, keys, ready_keys, request_timestamp,
                                                   request.client_city, url=request.META.get('HTTP_REFERER'))

    extra_segments = ExtraTrainSegment.limit(extra_segments, borders)
    ExtraTrainSegment.fill_titles(extra_segments)
    ExtraTrainSegment.correct_stations(extra_segments, point_from, point_to, uri=request.META.get('HTTP_REFERER'))

    segments = extra_segments.values()

    for segment in segments:
        segment.is_extra = True

    segments = [s for s in segments if not s.gone]
    blablacar_segment = next((s for s in segments if s.t_type.id == TransportType.BLABLACAR), None)

    if blablacar_segment:
        segments.remove(blablacar_segment)

    context = {
        'date': True,
        'search_type': None,
        'currency_info': currency_info,
        'point_from': point_from,
        'point_to': point_to,
        'segments': segments,
        'updates': updates,
        'ttype': ttype,
    }

    def create_tariff_update_info(cooked_info):
        national_version = request.NATIONAL_VERSION

        if cooked_info.t_type == 'plane' and ttype != 'all':
            places = plane_tariff_places(cooked_info.places, currency_info)
        else:
            places = tariffs_places(cooked_info.places, currency_info, national_version=national_version)

        min_tariff = None
        if len(cooked_info.places):
            cheap_place = min(cooked_info.places, key=lambda x: x.tariff)
            min_tariff = currency(cheap_place.tariff, currency_info, getattr(cheap_place, 'min', False)) or ''

        return {
            'minTariff': min_tariff.replace('&nbsp;', ' '),
            'places': places,
            'places_names': [{'value': dynamic_gettext(place.name).upper(),
                              'name': dynamic_gettext(place.name).upper()}
                             for place in cooked_info.places],
            'eticket': cooked_info.et_marker
        }

    timetable_block_class = search_tpl.TimetableBlock
    if request.NATIONAL_VERSION == 'tr':
        timetable_block_class = UntypedTimetableBlock

    data.update({
        'lines': timetable_block_class.ajax_lines(request, context, segments, search_type),
        'specialLines': {
            'blablacar': timetable_block_class.ajax_lines(request,
                                                          context,
                                                          [blablacar_segment],
                                                          search_type) if blablacar_segment else []
        },
        'updates': dict(
            (key, create_tariff_update_info(cooked)) for key, cooked in updates.items()
        )
    })

    return data


def extended_search_points(search_type, point_from, point_to):
    city_from = isinstance(point_from, Station) and point_from.settlement_id and point_from.settlement or point_from
    city_to = isinstance(point_to, Station) and point_to.settlement_id and point_to.settlement or point_to

    if city_from != city_to:
        point_list_from = PointList(city_from, [city_from])
        point_list_to = PointList(city_to, [city_to])

        point_list_from, point_list_to = process_points_lists(
            point_list_from, point_list_to, suburban=search_type == 'suburban'
        )
        city_from, city_to = point_list_from.point, point_list_to.point

        if city_from != point_from:
            yield city_from, point_to

        if city_to != point_to:
            yield point_from, city_to

        if city_from != point_from and city_to != point_to:
            yield city_from, city_to


def search_segments(point_from, point_to, departure_date, transport_types=None):
    extended_points = None

    segments, nears, service_types = search_routes(point_from, point_to,
                                                   departure_date=departure_date,
                                                   include_interval=False,
                                                   transport_types=transport_types)

    if not segments:
        for extended_point_from, extended_point_to in extended_search_points(transport_types, point_from, point_to):
            segments, nears, service_types = search_routes(extended_point_from, extended_point_to,
                                                           departure_date=departure_date,
                                                           include_interval=False,
                                                           transport_types=transport_types)

            if segments:
                extended_points = extended_point_from, extended_point_to

                break

    return segments, service_types, extended_points


def get_template(request, tcode, context):
    if tcode == 'train':
        return search_tpl.TrainTemplate.render(request, context)

    if tcode == 'suburban':
        return search_tpl.SuburbanTemplate.render(request, context)

    if tcode == 'plane':
        return search_tpl.PlaneTemplate.render(request, context)

    if tcode == 'bus':
        return search_tpl.BusTemplate.render(request, context)

    if tcode == 'helicopter':
        return search_tpl.HelicopterTemplate.render(request, context)

    if tcode in TransportType.WATER_TTYPE_CODES:
        return search_tpl.WaterTemplate.render(request, context)

    context['error'] = gettext(u'Указан неизвестный тип транпорта')

    return err_tpl.Template.render(request, context)


class SearchView(View):

    def search_segments(self, point_from, point_to, when, search_type):
        raise NotImplementedError

    def get_supplement(self, search_type, service_types):
        supplements = [ttype for ttype in ['train', 'plane', 'bus', 'helicopter'] if ttype in service_types]

        if search_type and search_type in supplements:
            supplements = [search_type]

        supplements.append('blablacar')

        return supplements

    def mark_extra_segments(self, segments):
        for segment in segments:
            if isinstance(segment, (ExtraTrainSegment, Segment)):
                segment.is_extra = True

    def build_search_context(self, request, search_type=None, allow_blablacar=True):
        # Строим контекст для сводной и типизированной выдачи

        form = SearchForm(request.GET.copy(), request=request, ttype=search_type)

        redirect_result = self.try_redirect_to_validate_form(request, form)
        if redirect_result:
            return redirect_result, None

        context = {
            'form': form
        }

        point_from = form.cleaned_data['from'].point
        point_to = form.cleaned_data['to'].point
        when = form['when'].data

        segments, service_types, extended_points = self.search_segments(point_from, point_to, when, search_type)

        if extended_points:
            context['extended'] = True
            context['requested_from'] = point_from
            context['requested_to'] = point_to

            point_from = extended_points[0]
            point_to = extended_points[1]

        context['point_from'] = point_from
        context['point_to'] = point_to
        context['page_title'] = format_page_title(point_from, point_to, when)

        currency_info = fetch_currency_info(request)

        context['currency_info'] = currency_info

        search_log(request=request,
                   search_type=search_type,
                   point_from=point_from,
                   point_to=point_to,
                   when=when,
                   segments=segments)

        for segment in segments:
            segment.display_info = DisplayInfo()

        supplements = self.get_supplement(search_type, service_types)
        allow_blablacar = allow_blablacar and check_blablacar_allowed(search_type, point_from, point_to)
        context['ajax_tariffs_info'] = \
            fill_tariffs_info(segments, currency_info, supplements, point_from, point_to, when, allow_blablacar, request)

        self.mark_extra_segments(segments)

        context['segments'] = segments

        context['teasers'] = TeaserSetTouch(request, 'search', {
            'points': [point_from, point_to],
            'routes': segments
        })

        context['show_avia_disclaimer'] = 'plane' in supplements

        return None, context

    def try_redirect_to_validate_form(self, request, form):

        def try_validate_form():
            if not form.is_valid():
                context = {
                    'form': form,
                    'page_title': gettext(u'Некорректные параметры поиска'),
                    'error': form.list_errors()
                }

                return err_tpl.Template.render(request, context)

        def try_choose_country_station():
            form_data = form.cleaned_data

            point_from = form_data['from'].point
            point_to = form_data['to'].point

            if not isinstance(point_from, Country) and not isinstance(point_to, Country):
                return

            context = {
                'form': form,
                'page_title': gettext(u'Уточните пункты отправления и прибытия')
            }

            if isinstance(point_from, Country):
                cities = get_biggest_cities(point_from)

                context['field'] = 'from'
                context['country'] = point_from

            else:
                cities = get_biggest_cities(point_to)

                context['field'] = 'to'
                context['country'] = point_to

            context['airports'] = cities

            if not cities:
                context['no_airports_error'] = True

                return search_tpl.Template.render(request, context)

            return precise_tpl.ChooseCountryStation.render(request, context)

        def try_precise_points():
            form_data = form.cleaned_data

            if not form_data['ambiguous']:
                return

            context = {
                'form': form,
                'type': 'search',
                'page_title': gettext(u'Уточните пункты отправления и прибытия')
            }

            if form_data['from'].has_variants():
                context['field'] = 'from'
                context['variants'] = form_data['from'].variants

            else:
                context['field'] = 'to'
                context['variants'] = form_data['to'].variants

            return precise_tpl.Template.render(request, context)

        for handler in [try_validate_form, try_choose_country_station, try_precise_points]:
            template_result = handler()
            if template_result:
                return template_result


class GroupedSearchView(SearchView):

    def search_segments(self, point_from, point_to, when, search_type):
        return search_segments(point_from=point_from,
                               point_to=point_to,
                               departure_date=when)

    def get(self, request):
        """Сводная выдача"""

        redirect, context = self.build_search_context(request)

        if redirect:
            return redirect

        groups = self.group_segments_by_transports(context['segments'])
        tcodes = groups.keys()

        if len(tcodes) == 1:  # Показываем типизированную выдачу
            tcode = tcodes[0]

            if tcode == 'aeroex':
                tcode = 'suburban'

            context['form'].ttype = tcode

            return TypedSearchView().get_with_context(request, tcode, context)

        context['groups'] = groups

        return search_tpl.Template.render(request, context)

    def group_segments_by_transports(self, segments):

        segments.sort(key=get_transport_code_from)

        groups = {}

        for code, segments in itertools.groupby(segments, key=get_transport_code_from):
            if code == 'blablacar':
                continue

            segments = list(segments)

            not_gone_segments = [s for s in segments if not s.gone]

            min_tariffs = []
            min_tariff = None

            for segment in not_gone_segments:
                tariffs_info = segment.display_info.get('tariffs_info')

                if not tariffs_info:
                    continue

                tariffs = [place.tariff for place in tariffs_info.places]

                if tariffs:
                    min_tariffs.append(min(tariffs))

            if min_tariffs:
                min_tariff = min(min_tariffs)

            groups[code] = {
                'cnt': len(segments),
                'duration': min(s.duration for s in segments),
                'min_tariff': min_tariff
            }

        return groups


class UntypedSearchView(GroupedSearchView):

    def get(self, request, *args, **kwargs):
        """Нетипизированная выдача"""

        redirect, context = self.build_search_context(request, allow_blablacar=False)

        if redirect:
            return redirect

        groups = self.group_segments_by_transports(context['segments'])

        context['groups'] = groups
        context['form'].ttype = 'all'
        context['end_stations_ids'] = fetch_end_station_ids([s.thread for s in context['segments'] if hasattr(s, 'thread')])

        return untypedSearch_tpl.UntypedTimetableTemplate.render(request, context)


class TypedSearchView(SearchView):

    def search_segments(self, point_from, point_to, when, search_type):

        if search_type in TransportType.WATER_TTYPE_CODES:
            transport_types = TransportType.WATER_TTYPE_CODES
        else:
            transport_types = (search_type,)

        return search_segments(point_from=point_from,
                               point_to=point_to,
                               departure_date=when,
                               transport_types=transport_types)

    def get(self, request, search_type):
        """Типизированная выдача"""

        redirect, context = self.build_search_context(request, search_type)

        if redirect:
            return redirect

        return self.get_with_context(request, search_type, context)

    def get_with_context(self, request, search_type, context):

        context['end_stations_ids'] = fetch_end_station_ids([s.thread for s in context['segments'] if hasattr(s, 'thread')])

        return get_template(request, search_type, context)


def search_log(request, search_type, point_from, point_to, when, segments):
    t_type_counts = {}

    for s in segments:
        thread = getattr(s, 'thread', None)

        if thread:
            t_type = thread.t_type

            if t_type:
                t_type_counts[t_type.code] = t_type_counts.get(t_type.code, 0) + 1

    log_data = {
        'from_id': point_from.point_key,
        'to_id': point_to.point_key,
        'transport_type': search_type,
        'when': when,
        'geoid': getattr(request, "geoid", None),
        'return_date': None,
        'klass': None,
        'adults': None,
        'children': None,
        'infants': None,
        't_type_counts': json.dumps(t_type_counts, separators=(',', ':')),
        'national_version': getattr(request, 'NATIONAL_VERSION', None)
    }

    search_logger.log(request, log_data)

    log_data['service'] = 'rasp-touch'
    log_data['tskv_format'] = 'rasp-users-search-log'
    log_data['referer'] = request.META.get('HTTP_REFERER')

    rasp_users_search_logger.log(request, log_data)
