# -*- coding: utf-8 -*-

from cgi import escape
from itertools import groupby

from django.http import HttpResponseRedirect, HttpResponsePermanentRedirect
from django.utils.translation import get_language

from common.models.transport import TransportType
from common.utils.avia import get_avia_thread_url
from common.views.thread_search import process_threads

from common.xgettext.i18n import gettext, xgettext
from common.views.tariffs import DisplayInfo
from travel.rasp.library.python.common23.date import environment
from common.utils.locations import composeurl
from route_search.by_number import find_threads_by_query
from route_search.shortcuts import search_routes

from travel.rasp.touch.touch.core.templates.search import threads as threads_tpl
from travel.rasp.touch.touch.core.templates.utils import NBSP
from travel.rasp.touch.forms import RouteSearchForm


def special_search_result(request, threads, context):
    query = context['query']
    context['number'] = threads[0].L_title_special()

    searches = list(set((thread.first_station, thread.last_station) for thread in threads))
    searches.sort(key=lambda pair: sorted([pair[0].settlement_id, pair[1].settlement_id]))

    directions = {}

    for point_from, point_to in searches:
        key = (point_from.settlement or point_from, point_to.settlement or point_to)

        directions.setdefault(key, []).append((point_from, point_to))

    search_results = []

    for direction, stations in directions.iteritems():
        segments_res = []

        for search_pair in stations:
            segments, _nears, _stypes = search_routes(search_pair[0], search_pair[1], include_interval=False)
            segments = [segment for segment in segments if segment.thread.L_title_special.contains(query)]

            segments_res.extend(segments)

        segments_res.sort(key=lambda s: s.departure.time())

        for segment in segments_res:
            segment.display_info = DisplayInfo()

        if get_language() == 'ru':
            title = [direction[0].L_title(case='phrase_from'), ' ', direction[1].L_title(case='phrase_to')]

        else:
            title = [direction[0].L_title(), NBSP, '&mdash; ', direction[1].L_title()]

        if segments_res:
            fragment = segments_res[0].L_title()

        else:
            fragment = u''

        search_results.append({
            'fragment': fragment,
            'segments': segments_res,
            'from_id': direction[0].id,
            'point_from': direction[0],
            'point_to': direction[1],
            'to_id': direction[1].id,
            'link_text': [direction[0].L_title(), u' — ', direction[1].L_title()],
            'title': title
        })

    context['search_results'] = search_results

    search_navigation = [list(v) for k, v in groupby(search_results,
                                                     key=lambda result: sorted([result['from_id'],
                                                                                result['to_id']]))]

    if len(search_navigation) == 1:
        context['search_navigation'] = map(lambda item: (item,), search_navigation[0])
    else:
        context['search_navigation'] = map(None, *search_navigation)

    return threads_tpl.SpecialSearchTemplate.render(request, context)


def ordinal_search_result(request, threads, context):
    query = context['query']

    # Исключаем рейсы МТА
    threads = [t for t in threads if t.supplier_id != 44]

    if len(threads) == 1:
        if threads[0].t_type_id == TransportType.PLANE_ID:
            return HttpResponsePermanentRedirect(get_avia_thread_url(
                threads[0], request.tld, utm_medium='flight_landing', utm_campaign='redirect301'
            ))
        return HttpResponseRedirect(composeurl('thread', kwargs={'uid': threads[0].uid}))

    t_code = request.GET.get('ttype')

    if not t_code:
        # Сводная выдача
        types_order = dict((v, k) for k, v in enumerate(('plane', 'train', 'suburban', 'bus')))

        threads.sort(key=lambda t: types_order.get(t.t_type.code, 99))

        # Группировка по видам транспорта
        t_type_groups = dict(
            (code, len(set([t.route_id for t in t_type_threads])))
            for code, t_type_threads in groupby(threads, key=lambda t: t.t_type.code)
        )

        t_codes = t_type_groups.keys()

        if len(t_codes) > 1:
            context['t_type_groups'] = t_type_groups

            return threads_tpl.Template.render(request, context)

        if t_codes:
            t_code = t_codes[0]

        else:
            t_code = None

    context['t_code'] = t_code

    threads = [t for t in threads if t.t_type.code == t_code]

    exact = None

    exact_threads = [t for t in threads if t.number.lower() == query.lower()]

    if exact_threads:
        exact = exact_threads[0]

        threads = [t for t in threads if t.route.id != exact.route.id]

    if exact and not threads:
        if exact.t_type_id == TransportType.PLANE_ID:
            return HttpResponsePermanentRedirect(get_avia_thread_url(
                exact, request.tld, utm_medium='flight_landing', utm_campaign='redirect301'
            ))
        return HttpResponseRedirect(composeurl('thread', kwargs={'uid': exact.uid}))

    context['exact'] = exact

    threads.sort(key=lambda t: (t.route.id, t.L_title(), t.L_title_special()))

    res_threads = []

    for rid, r_threads in groupby(threads, key=lambda t: t.route.id):

        t = min(r_threads, key=lambda t: t.first_station.departure_time)

        res_threads.append(t)

    context['threads'] = res_threads
    return threads_tpl.OrdinalSearchTemplate.render(request, context)


def main(request):
    form = RouteSearchForm(request.GET, request=request)

    context = {
        'page_title': gettext(u'Поиск рейсов'),
        'form': form,
        'threads': None
    }

    if not form.is_valid():
        return threads_tpl.Template.render(request, context)

    query = form['number'].data

    context['query'] = query

    context['page_title'] = xgettext(u'Поиск рейсов - <query/>', query=escape(query))

    result = find_threads_by_query(query, form.get_t_types_ids())

    threads = result.threads

    if not threads:
        return threads_tpl.Template.render(request, context)

    threads = process_threads(threads, environment.now_aware())

    if result.is_special_train:
        return special_search_result(request, threads, context)

    return ordinal_search_result(request, threads, context)
