# coding: utf-8

import logging
from datetime import datetime

from django.http import HttpResponseRedirect, Http404, HttpResponse, HttpResponsePermanentRedirect

from common.apps.suburban_events.api import get_states_for_thread
from common.models.schedule import RThread, StationSchedule
from common.models.transport import TransportType
from travel.rasp.library.python.common23.date import environment
from common.utils.avia import get_avia_thread_url
from common.utils.date import MSK_TZ
from common.utils.httpresponses import jsonp_response
from common.utils.locations import composeurl
from common.views.currency import fetch_currency_info
from common.views.tariffs import DisplayInfo
from common.views.thread import calc_thread_start_date, get_thread_departure_date, ThreadWithoutDaysError
from common.xgettext.i18n import xgettext, ngettext
from route_search.models import RThreadSegment
from travel.rasp.touch.forms import ThreadForm
from travel.rasp.touch.tariffs.views import fill_tariffs_info
from travel.rasp.touch.touch.core.helpers.thread import get_best_stations
from travel.rasp.touch.touch.core.lib.train_tariff_api_client import train_tariff_api_client, TrainTariffRecord
from travel.rasp.touch.touch.core.templates import thread as thread_tpl
from travel.rasp.touch.touch.core.templates.utils import currency


log = logging.getLogger(__name__)


def page_title(thread, rtstations):
    number = thread.number
    ttype = thread.t_type_id

    kwargs = {
        'number': number,
        'title': thread.L_title()
    }

    if thread.is_combined and number:
        title = xgettext(u'№<number/> <title/>', **kwargs)

    elif ttype == TransportType.BUS_ID:
        if thread.supplier.code == 'mta' and thread.number:
            title = xgettext(u"Маршрут автобуса №<number/> <title/>", **kwargs)

        else:
            title = xgettext(u"Маршрут автобуса <title/>", **kwargs)

    elif ttype in (TransportType.PLANE_ID, TransportType.HELICOPTER_ID):
        points_titles = []

        for rts in rtstations:
            station = rts.station
            settlement = station.settlement

            if settlement:
                points_titles.append(settlement.L_title())

            else:
                points_titles.append(station.L_title())

        kwargs['stations'] = u"&nbsp;&mdash;&nbsp;".join(points_titles)

        title = xgettext(u"Рейс <number/>, <stations/>", **kwargs)

    elif ttype == TransportType.WATER_ID:
        title = xgettext(u"Теплоход <title/>", **kwargs)

    else:
        title_special = thread.L_title_special()

        kwargs['special_title'] = title_special

        if thread.type.code == 'through_train':
            title = xgettext(u"Беспересадочный вагон <number/>, <title/>", **kwargs)

        elif title_special and thread.is_deluxe:
            if number:
                title = xgettext(u"Фирменный поезд «<special-title/>» <number/>, <title/>", **kwargs)

            else:
                title = xgettext(u"Фирменный поезд «<special-title/>», <title/>", **kwargs)

        elif thread.is_deluxe:
            if number:
                title = xgettext(u"Фирменный поезд <number/>, <title/>", **kwargs)

            else:
                title = xgettext(u"Фирменный поезд, <title/>", **kwargs)

        elif title_special:
            if number:
                title = xgettext(u"Поезд «<special-title/>» <number/>, <title/>", **kwargs)

            else:
                title = xgettext(u"Поезд «<special-title/>», <title/>", **kwargs)

        else:
            if number:
                title = xgettext(u"Поезд <number/>, <title/>", **kwargs)

            else:
                title = xgettext(u"Поезд <title/>", **kwargs)

    return title


def get_duration(rtstations, start_dt_aware, start_dt):
    if not rtstations:
        return

    last_rts = rtstations[-1]

    return last_rts.get_arrival_dt(start_dt) - start_dt_aware


def get_schedule_days(thread, rtstations, start_dt):
    if rtstations:
        shift = rtstations[0].calc_days_shift(event='departure', start_date=start_dt)

    else:
        shift = 0

    return thread.L_days_text(shift, except_separator=", ", html=True,
                              template_only=False, show_all_days=True)


def get_thread_context(request, uid):
    form = ThreadForm(request.GET.copy(), request=request)
    form.full_clean()

    if not form.is_valid():
        raise Http404

    context = form.cleaned_data.copy()
    form_data = form.cleaned_data

    thread = get_thread_or_fail(request, uid)
    if isinstance(thread, HttpResponse):
        return thread

    if form_data.get('departure_from') and form_data.get('station_from'):
        context['departure'] = calc_thread_start_date(
            thread, form_data['station_from'], form_data['departure_from']
        )

    try:
        departure_date = get_thread_departure_date(context['departure'], thread)
    except ThreadWithoutDaysError:
        raise Http404

    start_dt = datetime.combine(departure_date, thread.tz_start_time)
    start_dt_aware = thread.pytz.localize(start_dt)

    if thread.t_type_id == TransportType.PLANE_ID:
        return HttpResponsePermanentRedirect(get_avia_thread_url(
            thread, request.tld, day=departure_date if context['departure'] else None,
            utm_medium='flight_landing', utm_campaign='redirect301'
        ))

    context['start_dt'] = start_dt

    rtstations = list(thread.path.select_related('station', 'station__settlement',
                                                 'station__station_type', 'station__majority',
                                                 'terminal', 'departure_t_model'))

    if thread.is_aeroexpress or thread.is_express:
        rtstations = [rts for rts in rtstations if rts.tz_arrival != rts.tz_departure]

    station_from = form_data['station_from']
    station_to = form_data['station_to']

    rtstation_from, rtstation_to = get_best_stations(station_from, station_to, rtstations)

    thread_stops = None

    if station_from or rtstations:
        try:
            s = station_from or rtstations[0].station

            schedule = StationSchedule.objects.filter(thread=thread, station=s)[0]
            thread_stops = schedule.L_stops()

        except IndexError:
            pass

    if rtstation_from and rtstation_to and rtstation_from.id > rtstation_to.id:
            context['station_from'], context['station_to'] = station_to, station_from
            context['point_from'], context['point_to'] = context['point_to'], context['point_from']

    context['segment'] = None
    if (rtstations and station_from and station_to
            and form_data['point_from'] and form_data['point_to']):

        rts_from = rtstation_from or rtstations[0]
        rts_to = rtstation_to or rtstations[-1]

        loc_dep_dt_aware = rts_from.get_loc_departure_dt(start_dt)
        loc_arr_dt_aware = rts_to.get_loc_arrival_dt(start_dt)

        segment = build_segment(thread=thread,
                                loc_departure_dt=loc_dep_dt_aware,
                                loc_arrival_dt=loc_arr_dt_aware,
                                thread_start_date=start_dt,
                                station_from=station_from,
                                station_to=station_to,
                                rtstation_from=rtstation_from,
                                rtstation_to=rtstation_to)

        fill_tariffs_info(segments=[segment],
                          currency_info=fetch_currency_info(request),
                          supplement=[],
                          point_from=form_data['point_from'],
                          point_to=form_data['point_to'],
                          when=loc_dep_dt_aware.date(),
                          request=request,
                          allow_blablacar=False)

        context['display_info'] = segment.display_info
        context['segment'] = segment

    context['thread_state'], context['states_by_rtstation'] = get_states_for_thread(thread, departure_date, rtstations)
    context['currency_info'] = fetch_currency_info(request)
    context['thread'] = thread
    context['rtstations'] = rtstations
    context['rtstation_from'] = rtstation_from
    context['rtstation_to'] = rtstation_to
    context['thread_stops'] = thread_stops
    context['duration'] = get_duration(rtstations, start_dt_aware, start_dt)
    context['schedule_days'] = get_schedule_days(thread, rtstations, start_dt)
    context['company'] = thread.company
    context['page_title'] = page_title(thread, rtstations)

    return context


def main(request, uid):
    context = get_thread_context(request, uid)

    if isinstance(context, HttpResponse):
        return context

    t_type = context['thread'].t_type.code
    # пытаемся получить информацию для покупки билета
    buy_context = build_buy_context(
        request=request,
        currency_info=context['currency_info'],
        display_info=context.get('display_info'),
        thread=context['thread'],
        segment=context['segment']
    )

    if buy_context['state'] != 'error':
        context['buy_context'] = buy_context
        if t_type == 'plane':
            return thread_tpl.ThreadWithAviaBuy.render(request, context)
        if t_type == 'train':
            return thread_tpl.ThreadWithTrainBuy.render(request, context)
    return thread_tpl.Template.render(request, context)


def get_thread_or_fail(request, uid):
    try:
        thread = RThread.objects.all().select_related().get(uid=uid)

        if thread.type.code == 'cancel' or thread.route.hidden:
            raise Http404

        return thread

    except RThread.DoesNotExist:
        number = request.GET.get('number')
        if not number:
            if '_' in uid:
                number = uid.split('_')[0]
        if not number or number == 'empty':
            number = uid

        return HttpResponseRedirect(composeurl('search_threads', params={'number': number}))


@jsonp_response
def buy_info(request):
    currency_info = fetch_currency_info(request)

    thread_context = get_thread_context(request, request.GET['thread'])
    buy_context = train_build_buy_context(request, currency_info, thread_context['segment'], init_query=False)

    return buy_context


@jsonp_response
def avia_buy_info(request, uid):
    currency_info = fetch_currency_info(request)

    context_or_redirect = get_thread_context(request, uid)
    if isinstance(context_or_redirect, HttpResponse):
        raise Http404

    return plane_build_buy_context(request, currency_info, context.get('display_info'), context['thread'].uid)


def _find_cheapest_plane(display_info):
    if display_info is None:
        return

    tariff_info = display_info.get('tariffs_info')
    if not tariff_info or not tariff_info.places:
        return

    return min(tariff_info.places, key=lambda x: x.tariff)


def build_buy_context(request, currency_info, display_info, thread, segment):
    t_type = thread.t_type.code
    if t_type == 'train':
        try:
            return train_build_buy_context(request, currency_info, segment, init_query=True)
        except Exception as e:
            return {
                'state': 'error'
            }
    if t_type == 'plane':
        return plane_build_buy_context(request, currency_info, display_info, thread.uid)

    return {
        'state': 'error'
    }


def plane_build_buy_context(request, currency_info, display_info, uid):
    place = _find_cheapest_plane(display_info)
    if not place:
        return {
            'state': 'loading',
            't_type': 'plane',
            'update_url': composeurl('avia_buy_info', params=request.GET, kwargs={'uid': uid})
        }

    partner = place.partner

    return {
        'state': 'form',
        't_type': 'plane',
        'update_url': composeurl('avia_buy_info', params=request.GET, kwargs={'uid': uid}),
        'variant': {
            'tariff': currency(place.tariff, currency_info, getattr(place, 'min', False)),
            'deep_link': place.deep_link,
            'order_link': place.order_link,
            'partner_title': partner.title,
            'logo': partner.logo
        }
    }


def train_build_buy_context(request, currency_info, segment, init_query):
    context = {
        'state': 'error',
        't_type': 'train'
    }
    if segment is None:
        return context

    train_buy_model = train_tariff_api_client.find_tariffs_for(
        from_station=segment.station_from,
        to_station=segment.station_to,
        departure_dt=segment.departure,
        number=segment.number,
        national_version=request.NATIONAL_VERSION,
        rates=currency_info.rates,
        init_query=init_query
    )

    if train_buy_model.querying is False and len(train_buy_model.records) == 0:
        return context

    context.update({
        'buy_info_url': composeurl('buy_info', params=request.GET),
        'partner_site': 'www.ufs-online.ru',
        'train_order_url_owner': train_buy_model.train_order_url_owner
    })

    context['state'] = 'many-classes'

    if train_buy_model.querying is True and len(train_buy_model.records) == 0:
        context['state'] = 'lazy-init'
        return context

    if len(train_buy_model.records) == 1:
        context['state'] = 'single-class'

    context['variants'] = [{
        'result_text': ''.join(build_result_text(record)),
        'code': record.coach_type,
        'tariff': currency(record.price, currency_info, _from=True),
        'name': record.coach_title,
        'train_order_url': record.train_order_url
    } for record in train_buy_model.records]

    context['selected_item'] = context['variants'][0]

    return context


def build_result_text(class_row):
    # type: (TrainTariffRecord) -> unicode

    common = class_row.seats
    up = class_row.upper_seats
    down = class_row.lower_seats

    if not down and not up:
        return ngettext(common, u'%d место', u'%d места', u'%d мест') % common
    else:
        return xgettext(u'верхних — <up/>, нижних — <down/>.',
                        up=str(up),
                        down=str(down))


def build_segment(thread, loc_departure_dt, loc_arrival_dt, thread_start_date, station_from, station_to, rtstation_from, rtstation_to):
    segment = RThreadSegment()

    segment.station_from = station_from
    segment.station_to = station_to

    segment.thread = thread

    segment.departure = loc_departure_dt
    segment.arrival = loc_arrival_dt

    segment.start_date = thread_start_date

    segment.rtstation_from = rtstation_from
    segment.rtstation_to = rtstation_to

    segment.stops_translations = []

    segment.now = MSK_TZ.localize(environment.now())

    segment.display_info = DisplayInfo()

    segment._init_data()

    return segment
