# -*- coding: utf-8 -*-

from __future__ import absolute_import

from django.db.models import Q
from django.utils.translation import get_language

from common.models.schedule import RTStation
from common.models.transport import TransportType
from common.utils.date import human_date_without_year
from common.xgettext.i18n import gettext, xgettext
from common.models.currency import Price

from travel.rasp.touch.touch.utils import station_ttype_codes


class NoDefaultProvided(object):
    pass


def get_attr_by_path(obj, path, default=NoDefaultProvided):
    """
    Same as getattr(), but allows dot notation lookup

    Examples:

    get_attr_by_path(int, '__class__.__name__') //type
    get_attr_by_path(int, '__class__.__name__') //<type 'type'>
    get_attr_by_path(int, 'a', None)

    """
    try:
        return reduce(getattr, path.split("."), obj)
    except Exception, e:
        if default != NoDefaultProvided:
            return default
        raise AttributeError()


def fetch_end_station_ids(threads):
    ids = {}

    rtstations = (RTStation.objects
                            .filter(thread__in=threads)
                            .filter(Q(tz_departure__isnull=True) |
                                    Q(tz_arrival__isnull=True))
                            .select_related('station'))

    for rtstation in rtstations:
        thread = rtstation.thread

        ids.setdefault(thread, {'arrival': (), 'departure': ()})

        if rtstation.tz_departure is None:
            ids[thread]['arrival'] = (rtstation.station.id, rtstation.station.settlement_id)

        if rtstation.tz_arrival is None:
            ids[thread]['departure'] = (rtstation.station.id, rtstation.station.settlement_id)

    return ids


def type_choices(station):
    codes = station_ttype_codes(station)

    type_names = {
        'tablo': gettext(u'табло'),
        'plane': gettext(u'самолёты'),
        'suburban': gettext(u'электрички'),
        'aeroex': gettext(u'аэроэкспрессы'),
        'train': gettext(u'поезда'),
        'schedule': gettext(u'расписание'),
        'sea': gettext(u'теплоходы'),
        'water': gettext(u'теплоходы'),
    }

    return [(c, type_names.get(c, gettext(u'Неизвестное направление'))) for c in codes]


def price_from_json(json_data):
    price = Price(value=json_data['value'], currency=json_data.get('currency'))
    price.base_value = json_data.get('baseValue')

    return price


def format_page_title(point_from, point_to, when=None, ttype=None):
    gettext_params = {
        'point_from_title': point_from.L_title,
        'point_to_title': point_to.L_title
    }

    if when:
        weekday_name = [
            gettext(u'в понедельник'),
            gettext(u'во вторник'),
            gettext(u'в среду'),
            gettext(u'в четверг'),
            gettext(u'в пятницу'),
            gettext(u'в субботу'),
            gettext(u'в воскресенье'),
        ][when.weekday()]

        gettext_params['when'] = [human_date_without_year(when) or u'', u', ', weekday_name]
        gettext_params['comma'] = ', '
    else:
        gettext_params['when'] = None
        gettext_params['comma'] = None

    ttype_to_human = {
        'train': gettext(u'поездов'),
        'bus': gettext(u'автобусов'),
        'suburban': gettext(u'электричек'),
        'plane': gettext(u'самолётов'),
        'water': gettext(u'теплоходов'),
        'sea': gettext(u'океанских лайнеров'),
        'river': gettext(u'кораблей'),
    }

    transport = ttype_to_human.get(ttype, None)

    if not transport and get_language() != 'tr':
        transport = gettext(u'рейсов')

    gettext_params['transport'] = transport

    return xgettext(u'Расписание <transport/>: <point-from-title/>&nbsp;&mdash; <point-to-title/><comma/><when/>', **gettext_params)


def get_transport_code_from(segment):
    code = segment.thread.subtype_code if segment.thread else segment.t_type.code

    if code in TransportType.WATER_TTYPE_CODES:
        return 'water'

    return code

