# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from django.apps import AppConfig

from common.db.mongo.base import ensure_indexes
from common.dev_tools.build import is_building
from travel.rasp.library.python.common23.date.environment import is_migration_allowed
from common.workflow import registry


class TrainApiAppConfig(AppConfig):
    name = 'travel.rasp.train_api'
    label = 'train_api'

    def ready(self):
        # в случае если мы собираемся - нам не нужно инициализировать процессы, celery и тд,
        # так как в некоторых случаях потребуются соединение до mongo
        if is_building():
            return

        from travel.rasp.train_api.celery import app  # noqa
        from travel.rasp.train_api.train_purchase.core.models import TrainOrder, TrainRefund, Payment
        from travel.rasp.train_api.train_purchase.workflow.booking import train_booking_scheme, TRAIN_BOOKING_PROCESS
        from travel.rasp.train_api.train_purchase.workflow.payment import payment_scheme, PAYMENT_PROCESS
        from travel.rasp.train_api.train_purchase.workflow.ticket_refund import TICKET_REFUND_PROCESS, ticket_refund_scheme

        registry.add_process(PAYMENT_PROCESS, payment_scheme, Payment)
        registry.add_process(TRAIN_BOOKING_PROCESS, train_booking_scheme, TrainOrder)
        registry.add_process(TICKET_REFUND_PROCESS, ticket_refund_scheme, TrainRefund)

        # чтобы защититься от http://api.mongodb.com/python/current/faq.html#is-pymongo-fork-safe
        if is_migration_allowed():
            self.ensure_mongo_indexes()

        # pre import urls
        # После форка, коммон импортирован, а часть train_api нет.
        # При выкатке пакета, до рестарта всего gunicorn, может рестартовать воркер по max_requests
        # и получить новую train_api при старом common.
        # Поэтому нужно загрузить train_api до форка, urls тянет все вьюхи, нам этого достаточно.
        import travel.rasp.train_api.urls_async  # noqa

    def ensure_mongo_indexes(self):
        from travel.rasp.train_api.train_purchase.core import models
        ensure_indexes(models, additional_allowed_base_classes=(models.TrainPurchaseBaseDocument,))
