# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import base64
import json
import threading

import mock
import pytest
from django.core.cache import cache
from django.test import override_settings, Client

from travel.rasp.library.python.common23.tester.full_settings.default_conftest import *  # noqa
from common.tester.transaction_context import transaction_fixture


def pytest_sessionstart():
    import travel.rasp.train_api.celery.worker  # noqa


@pytest.hookimpl(hookwrapper=True, tryfirst=True)
def pytest_runtest_setup(item):
    from travel.rasp.train_api.tariffs.train.base.utils import get_point_express_code

    get_point_express_code.cache_clear()

    yield


@pytest.yield_fixture()
def async_urlconf_client():
    with override_settings(ROOT_URLCONF='travel.rasp.train_api.urls_async'):
        yield Client()


@pytest.fixture(scope='module')
@transaction_fixture
def full_tariff_info(request):
    from common.apps.train.models import TariffInfo

    return TariffInfo.objects.create(
        code=TariffInfo.FULL_CODE, title_ru='Полный',
        ufs_request_code='1', ufs_response_codes='ПОЛНЫЙ, 72 ADULT, 71 ВЗР/ADULT, 71 ADULT',
        im_request_code='Full', im_response_codes='Full'
    )


@pytest.fixture
def worker_cache_stub():
    from travel.rasp.train_api.tariffs.train.base import worker

    with override_settings(CACHES={'default': {'BACKEND': 'django.core.cache.backends.locmem.LocMemCache'}}), \
            mock.patch.object(worker, 'global_cache_sync_add',
                              side_effect=lambda x, y, z: cache.add(x, y, z)) as m_global_cache_sync_add, \
            mock.patch.object(worker, 'global_cache_sync_set',
                              side_effect=lambda x, y, z: cache.set(x, y, z)) as m_global_cache_sync_set, \
            mock.patch.object(worker, 'global_cache_sync_delete',
                              side_effect=lambda x: cache.delete(x)) as m_global_cache_sync_delete, \
            mock.patch.object(cache, 'm_global_cache_sync_add', m_global_cache_sync_add, create=True), \
            mock.patch.object(cache, 'm_global_cache_sync_set', m_global_cache_sync_set, create=True), \
            mock.patch.object(cache, 'm_global_cache_sync_delete', m_global_cache_sync_delete, create=True):
        yield cache
        cache.clear()


def build_worker_stub():
    class WorkerStub(threading.Thread):
        call_count = 0

        def start(self):
            WorkerStub.call_count += 1
            self.run()

        def run(self):
            super(WorkerStub, self).run()

        def start_daemon(self):
            self.daemon = True
            self.start()

    return WorkerStub


class StubInfo():
    def __init__(self):
        self.call_count = 0


@pytest.fixture
def worker_stub():
    from travel.rasp.train_api.tariffs.train.base.worker import Worker, PoolWorker

    mockinfo = StubInfo()

    def mock_run(self):
        super(Worker, self).run()

    def mock_start(self):
        mockinfo.call_count += 1
        self.run()

    def mock_pool_start(self):
        mockinfo.call_count += 1
        self.target(*self.args)

    with mock.patch.object(Worker, 'run', mock_run), \
            mock.patch.object(Worker, 'start', mock_start), \
            mock.patch.object(PoolWorker, 'start', mock_pool_start):
        yield mockinfo


@pytest.fixture()
def backoffice_client(async_urlconf_client):
    from common.apps.train_order.models import BackofficeUser

    user = BackofficeUser.objects.create(username='yndx.employee', is_active=True, is_admin=True)
    auth_header = base64.b64encode(json.dumps({'type': 'direct', 'login': user.username}))
    async_urlconf_client.defaults['HTTP_AUTHORIZATION'] = auth_header
    async_urlconf_client.user = user
    try:
        yield async_urlconf_client
    finally:
        user.delete()


@pytest.fixture(scope='module')
def mock_trust_client():
    from travel.rasp.train_api.train_purchase.core import models

    with mock.patch.object(models, 'TrustClient', autospec=True) as m_trust_client:
        m_instance = m_trust_client.return_value
        m_instance.get_receipt_url.return_value = None
        m_instance.get_receipt_clearing_url.return_value = None
        m_instance.get_refund_receipt_url.return_value = None
        yield m_instance


@pytest.fixture(autouse=True)
def clear_active_partners_cache():
    from travel.rasp.train_api.train_purchase.core.enums import clear_is_partner_active_cache
    clear_is_partner_active_cache()


@pytest.fixture()
def protobuf_data_provider():
    from travel.library.python.dicts.trains.station_express_alias_repository import StationExpressAliasRepository
    from travel.rasp.train_api.pb_provider import PROTOBUF_DATA_PROVIDER

    original_repos = vars(PROTOBUF_DATA_PROVIDER)
    PROTOBUF_DATA_PROVIDER.alias_repo = StationExpressAliasRepository(PROTOBUF_DATA_PROVIDER.alias_repo._index_field)

    yield PROTOBUF_DATA_PROVIDER

    for prop, repo in original_repos.iteritems():
        setattr(PROTOBUF_DATA_PROVIDER, prop, repo)
