# -*- coding: utf-8 -*-
from __future__ import unicode_literals, absolute_import, division, print_function

from common.models.geo import Settlement, Station, CityMajority
from common.models.transport import TransportType


def get_connected_stations(settlement, max_majority):
    qs = Station.objects.filter(hidden=False, majority__id__lt=max_majority)
    if settlement.is_foreign():
        qs = qs.filter(t_type_id=TransportType.PLANE_ID)

    stations = set(qs.filter(settlement=settlement))
    stations.update(qs.filter(station2settlement__settlement=settlement))

    return stations


def get_related_stations(settlement, max_majority):
    qs = Station.objects.filter(hidden=False, majority__id__lt=max_majority)
    if settlement.is_foreign():
        qs = qs.filter(t_type_id=TransportType.PLANE_ID)

    return qs.filter(related_settlements__settlement=settlement)


def get_visible_settlement_by_id(id):
    """
    Получает нескрытый город по id.
    Если по данному id город скрыт, то возвращает None.
    """
    try:
        settlement = Settlement.objects.get(id=id)
    except Settlement.DoesNotExist:
        return None

    if not settlement.hidden:
        return settlement

    return None


def get_visible_nearest_settlement_by_geo_ids(geo_ids, national_version):
    """
    Получает нескрытый населенный пункт по списку гео-идентификаторов.
    :param geo_ids: Обратный иерархический список гео-идентифиаторов населенных пунктов.
    :param national_version: Нац. домен, например 'ua' или 'ru'.
    Пример иерархии населенных пунктов для поселка Лобва (Новолялинский район, Свердловская область):
    [Лобва, Новая Ляля, Екатеринбург, Москва]
    """

    hidden_common_settlement = None
    hidden_region_capital = None
    hidden_country_capital = None

    # Поднимаемся вверх по иерархии городов и ищем первый не скрытый город
    for geo_id in geo_ids:
        try:
            settlement = Settlement.objects.get(_geo_id=geo_id)
        except Settlement.DoesNotExist:
            settlement = None

        if settlement:
            if not settlement.hidden:
                return settlement

            if settlement.majority_id > CityMajority.REGION_CAPITAL_ID:
                district_capital = get_visible_district_capital(settlement)
                if district_capital:
                    return district_capital
                hidden_common_settlement = settlement

            elif settlement.majority_id == CityMajority.REGION_CAPITAL_ID:
                hidden_region_capital = settlement
            elif settlement.majority_id == CityMajority.CAPITAL_ID:
                hidden_country_capital = settlement

    # Не нашли нескрытых населенных городов - ищем столицу региона
    if hidden_common_settlement and not hidden_region_capital:
        region_capital = get_visible_region_capital(hidden_common_settlement)
        if region_capital:
            return region_capital

    # Не нашли стлицу региона - ищем столицу страны
    last_hidden_settlement = hidden_region_capital or hidden_common_settlement
    if (last_hidden_settlement) and not hidden_country_capital:
        country_capital = get_visible_country_capital(last_hidden_settlement, national_version)
        if country_capital:
            return country_capital

    return None


def get_moscow():
    return Settlement.objects.get(id=Settlement.MOSCOW_ID)


def get_visible_nearest_capital(settlement, national_version):
    """
    Получает ближайшую нескрытую столицу - района, региона или страны
    """
    district_capital = get_visible_district_capital(settlement)
    if district_capital:
        return district_capital

    region_capital = get_visible_region_capital(settlement)
    if region_capital:
        return region_capital

    return get_visible_country_capital(settlement, national_version)


def get_visible_district_capital(settlement):
    """
    Получает нескрытую столицу района
    """
    if settlement.district and settlement.district.settlement and not settlement.district.settlement.hidden:
        return settlement.district.settlement

    return None


def get_visible_region_capital(settlement):
    """
    Получает нескрытую столицу региона
    """
    if settlement.region:
        capital = settlement.region.get_capital()
        if capital and not capital.hidden:
            return capital

    return None


def get_visible_country_capital(settlement, national_version):
    """
    Получает нескрытую столицу страны. Определение страны - в зависимоти от национальной версии.
    """
    country = settlement.translocal_country(national_version)
    if country:
        capital = country.get_capital()
        if capital and not capital.hidden:
            return capital

    return None
