# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from rest_framework import status
from rest_framework.response import Response
from rest_framework.views import exception_handler

from common.data_api.billing.trust_client import TrustClientException
from common.data_api.yasms.client import YaSMSClientError
from travel.rasp.train_api.train_partners.base import PartnerError
from travel.rasp.train_api.train_partners.im.base import ImNonCriticalError, ImError
from travel.rasp.train_api.train_purchase.core.models import ErrorInfo
from travel.rasp.train_api.train_purchase.serialization import ErrorInfoSchema

log = logging.getLogger(__name__)


YASMS_USER_ERROR_DESCRIPTION = {
    'LIMITEXCEEDED': 'Превышен дневной лимит отправки сообщений на телефонный номер.',
    'PERMANENTBLOCK': 'Номер заблокирован',
    'PHONEBLOCKED': 'Номер заблокирован',
}

YASMS_USER_ERROR_DESCRIPTION_DEFAULT = 'Ошибка отправки СМС, повторите позже или обратитесь в поддержку.'


def build_error_info(exc):
    """
    В зависимости от типа исключения, создаем объект ErrorInfo.
    :param exc: Исключение
    :type exc: Exception
    :rtype: ErrorInfo
    """
    if isinstance(exc, PartnerError):
        return ErrorInfo(
            type=ErrorType.PARTNER_ERROR,
            message=exc.get_user_message(),
            data=exc.get_data()
        )
    elif isinstance(exc, YaSMSClientError):
        return ErrorInfo(
            type=ErrorType.YASMS_ERROR,
            message=YASMS_USER_ERROR_DESCRIPTION.get(exc.err_code, YASMS_USER_ERROR_DESCRIPTION_DEFAULT),
            data={
                'err_code': exc.err_code,
                'err_msg': exc.err_msg
            }
        )
    elif isinstance(exc, TrustClientException):
        return ErrorInfo(
            type=ErrorType.TRUST_ERROR,
            message=exc.msg,
            data={}
        )
    else:
        return ErrorInfo(type=ErrorType.INTERNAL_ERROR)


class ErrorType(object):
    INTERNAL_ERROR = 'internal_error'
    PARTNER_ERROR = 'partner_error'
    YASMS_ERROR = 'yasms_error'
    SMS_VALIDATION_ERROR = 'sms_validation_error'
    PROCESS_EXCEPTION_STATE = 'process_exception_state'
    TRUST_ERROR = 'trust_error'


def rest_framework_exception_handler(exc, context):
    response = exception_handler(exc, context)
    if response is not None:
        return response

    log.exception('Ошибка обработки запроса:\n%r\n', context)

    error_info = build_error_info(exc)
    status_code = status.HTTP_400_BAD_REQUEST if isinstance(exc, ImNonCriticalError) \
        else status.HTTP_502_BAD_GATEWAY if isinstance(exc, ImError) \
        else status.HTTP_500_INTERNAL_SERVER_ERROR
    return render_error(error_info, status_code)


def render_error(error_info, http_status):
    return Response({
        'errors': {
            error_info.type: ErrorInfoSchema().dump(error_info).data,
        }
    }, status=http_status)
