import logging
import time

from django.conf import settings
from django.core.cache import cache
from rasp_vault.api import get_secret
from redis.sentinel import Sentinel

from common.settings.configuration import Configuration
from common.settings.utils import define_setting, bool_converter

log = logging.getLogger(__name__)

define_setting('RPS_LIMITER_REDIS_HOSTS', {
    Configuration.PRODUCTION: [
        'man-bp05feel4ojn05o6.db.yandex.net',
        'sas-cby6l2k5trbu9zdl.db.yandex.net',
        'vla-jdzr8z1ioxn15a5c.db.yandex.net'
    ]
}, default=['man-upd00sddcy5w161j.db.yandex.net', 'sas-h9s1noy6qfonekmc.db.yandex.net'])

define_setting('RPS_LIMITER_REDIS_SERVICE_NAME', {
    Configuration.PRODUCTION: 'raas__production__http_proxy_cache'
}, default='raas__testing__http_proxy_cache')

define_setting('RPS_LIMITER_REDIS_PASSWORD_SECRET', {
    Configuration.PRODUCTION: 'rasp-http-proxy-cache-production.REDIS_HTTP_PROXY_CACHE_PASSWORD'
}, default='rasp-http-proxy-cache-testing.REDIS_HTTP_PROXY_CACHE_PASSWORD')

define_setting('RPS_LIMITER_USE_DEFAULT_CACHE', default=False, converter=bool_converter)

RPS_LIMITER_REDIS_PORT = 26379
RPS_LIMITER_IM_CARPRICING_KEY = 'train_im_carpricing_ts_list'
RPS_LIMITER_IM_SEARCH_KEY = 'train_im_searchtariffs_ts_list'


class RpsLimiter(object):
    @property
    def _redis_sentinel(self):
        sentinel_hosts = [
            (h, RPS_LIMITER_REDIS_PORT)
            for h in settings.RPS_LIMITER_REDIS_HOSTS
        ]
        return Sentinel(
            sentinel_hosts, socket_timeout=0.5
        )

    @property
    def _redis_password(self):
        return get_secret(settings.RPS_LIMITER_REDIS_PASSWORD_SECRET)

    @property
    def _redis_master(self):
        return self._redis_sentinel.master_for(settings.RPS_LIMITER_REDIS_SERVICE_NAME, password=self._redis_password)

    @property
    def _redis_slave(self):
        return self._redis_sentinel.slave_for(settings.RPS_LIMITER_REDIS_SERVICE_NAME, password=self._redis_password)

    def get_slave(self):
        if settings.RPS_LIMITER_USE_DEFAULT_CACHE:
            return cache.client.slave
        return self._redis_slave

    def get_master(self):
        if settings.RPS_LIMITER_USE_DEFAULT_CACHE:
            return cache.client.master
        return self._redis_master

    def check_query_allowed(self, key, rps_limit, period):
        limit = int(rps_limit * period)
        try:
            max_queries_ts = self.get_slave().lindex(key, -limit)
            query_allowed = (max_queries_ts is None) or (float(max_queries_ts) + period < time.time())
            return query_allowed
        except Exception as ex:
            log.exception('Error check query allowed for key=%s', key)
        return True

    def save_query_token(self, *keys):
        now = time.time()
        try:
            for key in keys:
                self.get_master().rpush(key, now)
        except Exception as ex:
            log.exception('Error push query token for keys=%s', keys)

    def clean_tokens(self, key, rps_limit, period):
        try:
            limit = int(rps_limit * period)
            self.get_master().ltrim(key, -limit, -1)
        except Exception as ex:
            log.exception('Error clean tokens for keys=%s', key)


rps_limiter = RpsLimiter()
