# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import argparse
import sys
from datetime import datetime, timedelta

from bson import ObjectId
from django.core.management import BaseCommand

from travel.rasp.library.python.common23.date import environment
from common.utils.date import UTC_TZ, MSK_TZ
from common.utils.unicode_csv import UnicodeDictWriter
from travel.rasp.train_api.train_purchase.core.enums import OrderStatus
from travel.rasp.train_api.train_purchase.core.models import TrainOrder

REPORT_FIELDS = (
    'order_uid',
    'order_status',
    'purchase_token',
    'payment_status',
    'payment_resp_code',
    'payment_resp_desc'
)


def _to_utc(msk_naive_dt):
    return MSK_TZ.localize(msk_naive_dt).astimezone(UTC_TZ).replace(tzinfo=None)


def payment_info_iter(date, statuses):
    uts_start = _to_utc(datetime.combine(date, datetime.min.time()))
    uts_end = _to_utc(datetime.combine(date + timedelta(days=1), datetime.min.time()))
    order_qs = TrainOrder.objects.filter(
        id__gte=ObjectId.from_datetime(uts_start),
        id__lt=ObjectId.from_datetime(uts_end),
        status__in=statuses
    )
    for order in order_qs:
        for payment in order.payments:
            if payment.purchase_token:
                yield {
                    'order_uid': order.uid,
                    'order_status': order.status.value,
                    'purchase_token': payment.purchase_token,
                    'payment_status': payment.status,
                    'payment_resp_code': payment.resp_code,
                    'payment_resp_desc': payment.resp_desc
                }


def get_payment_info_xls(stream, date, statuses):
    writer = UnicodeDictWriter(stream, fieldnames=REPORT_FIELDS, dialect='excel-tab', encoding='utf-8')
    writer.writehead()
    for data in payment_info_iter(date, statuses):
        writer.writerow(data)


class Command(BaseCommand):
    help = 'Сбор статистики по платежам'

    def add_arguments(self, parser):
        all_statuse = [s.value for s in OrderStatus]
        parser.add_argument('--date', type=lambda d: datetime.strptime(d, "%Y-%m-%d"),
                            default=environment.today(), help='Дата')
        parser.add_argument('--statuses', type=OrderStatus, nargs='*', help='Статусы {}'.format(all_statuse),
                            default=all_statuse)
        parser.add_argument('--outfile', nargs='?', type=argparse.FileType('w'), default=sys.stdout,
                            help='Файл результата')

    def handle(self, outfile, date, statuses, **_extra):
        get_payment_info_xls(outfile, date, statuses)
