# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from rest_framework import status
from rest_framework.decorators import api_view
from rest_framework.response import Response

from common.models.geo import Settlement
from common.models.transport import TransportType
from travel.rasp.train_api.data_layer.train_popular_directions import get_train_popular_directions
from travel.rasp.train_api.popular_directions.serialization import PopularDirectionsSchema
from travel.rasp.train_api.popular_directions.settlement_images import get_settlement_image_urls


def _get_settlement_by_geo_id(geo_id):
    try:
        return Settlement.hidden_manager.get(_geo_id=geo_id) if geo_id else None
    except Settlement.DoesNotExist:
        return None


@api_view(['GET'])
def train_popular_directions(request):
    """
    Получает популярные направления поездов.
    ---
    parameters:
      - in: query
        name: query_params
        schema:
            type: object
            properties:
                geoId:
                    type: integer
                    description: geoId города
    responses:
        200:
            description: популярные направления для указанного города
            schema:
                $ref: 'PopularDirectionsSchema'
        404:
            description: город не найден
    """
    from common.data_api.min_prices.api import min_price_storage

    client_city = _get_settlement_by_geo_id(request.query_params.get('geoId'))
    if client_city is None:
        return Response(
            {'errors': {'geoId': 'Invalid value. Settlement was not found.'}},
            status=status.HTTP_404_NOT_FOUND
        )

    popular_directions = get_train_popular_directions(client_city)
    departure_settlement, arrival_settlements = (
        popular_directions['departure_city'], popular_directions['arrival_cities']
    )

    return Response(PopularDirectionsSchema(context={
        'best_offers': min_price_storage.find_best_offers(
            t_type=TransportType.objects.get(id=TransportType.TRAIN_ID),
            departure_settlement=departure_settlement,
            arrival_settlements=arrival_settlements
        ),
        'image_urls': get_settlement_image_urls(arrival_settlements)
    }).dump(popular_directions).data)
