# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import socket
from datetime import datetime
from time import sleep

from mongoengine import Q

from common.utils import gen_hex_uuid
from travel.rasp.library.python.common23.date import environment
from travel.rasp.train_api.train_partners.base.get_order_info import get_order_info
from travel.rasp.train_api.train_purchase.core.enums import InsuranceStatus
from travel.rasp.train_api.train_purchase.core.models import TrainRefund, RefundStatus, RefundUserInfo, TrainOrder

log = logging.getLogger()


def _get_external_refunds(start_date=None):
    filters = Q(is_external=True)
    if start_date:
        filters &= Q(created_at__gte=start_date)
    refunds = TrainRefund.objects.filter(filters)
    return refunds


def _is_order_with_insurance(order):
    return order.insurance and order.insurance.status == InsuranceStatus.CHECKED_OUT


def _get_insurances_refunded_by_im(order):
    order_info = get_order_info(order)
    insurance_by_operation_id = {i.operation_id: i for i in order.iter_insurances()}
    result = [insurance_by_operation_id[i.operation_id] for i in order_info.insurances if i.refund_amount]
    return result


def get_orders_with_not_refunded_insurances():
    external_refunds = _get_external_refunds(start_date=datetime(2019, 9, 20))
    external_refunds_with_insurance = [r for r in external_refunds if _is_order_with_insurance(r.order)]
    print(len(external_refunds_with_insurance))

    orders_with_not_refunded_insurances = {}
    current = 0
    length = len(external_refunds_with_insurance)
    for refund in external_refunds_with_insurance:
        current += 1
        insurances_refunded_by_im = _get_insurances_refunded_by_im(refund.order)
        log.info('{}/{}: {}'.format(current, length, refund.order.uid))
        if insurances_refunded_by_im:
            if all(i.refund_uuid for i in insurances_refunded_by_im):
                log.info('Ok')
            else:
                operation_ids = [i.operation_id for i in insurances_refunded_by_im if not i.refund_uuid]
                orders_with_not_refunded_insurances[refund.order_uid] = operation_ids
                log.info('Problem with {}'.format(', '.join(operation_ids)))
        sleep(5)

    return orders_with_not_refunded_insurances


def create_refund_payment_for_insurance(order_uid, insurance_operation_ids):
    order = TrainOrder.objects.get(uid=order_uid)
    refunded_insurance_ids = []
    for refund in order.iter_refunds():
        refunded_insurance_ids.extend(refund.insurance_ids)

    not_refunded_insurance_ids = set(insurance_operation_ids) - set(refunded_insurance_ids)
    if not not_refunded_insurance_ids:
        log.info('По {} уже проведен возврат.'.format(insurance_operation_ids))
        return None

    refund = TrainRefund.objects.create(
        uuid=gen_hex_uuid(),
        order_uid=order.uid,
        is_active=True,
        status=RefundStatus.NEW,
        blank_ids=None,
        insurance_ids=list(not_refunded_insurance_ids),
        user_info=RefundUserInfo(ip='127.0.0.1', uid=socket.gethostname(), region_id=213),
        is_external=True,
        created_at=environment.now_utc(),
    )
    log.info('Создан возврат {} для {}.'.format(refund.uuid, ', '.join(not_refunded_insurance_ids)))

    update_spec = {}
    for i, passenger in enumerate(order.passengers):
        if (passenger.insurance and passenger.insurance.operation_id in not_refunded_insurance_ids
                and passenger.insurance.trust_order_id):
            update_spec['set__passengers__{}__insurance__refund_uuid'.format(i)] = refund.uuid
    if update_spec:
        order.update(**update_spec)

    return refund
