# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging  # noqa
from datetime import datetime, timedelta
from decimal import Decimal, ROUND_HALF_UP
from io import BytesIO

import unicodecsv
from django.conf import settings
from mongoengine.queryset.visitor import Q

from common.db.mds.clients import mds_s3_public_client
from common.dynamic_settings.default import conf
from common.email_sender import guaranteed_send_email
from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from travel.rasp.library.python.common23.date import environment
from travel.rasp.library.python.common23.date.environment import now_utc
from common.utils.try_hard import try_hard
from travel.rasp.train_api.train_purchase.core.enums import OrderStatus
from travel.rasp.train_api.train_purchase.core.models import TrainOrder

log = logging.getLogger(__name__)

DEPTH_OF_EXPORT_IN_DAYS = 90
CHUNK_SIZE_IN_ROWS = 500
DATETIME_FORMAT = '%Y-%m-%dT%H:%M:%S'
CSV_HEADER = """Parameters:TimeZone=Etc/GMT,,,,
Google Click ID,Conversion Name,Conversion Time,Conversion Value,Conversion Currency
"""

define_setting('ERROR_EXPORT_DATA_FOR_GOOGLE_ADS_CAMPAIGN', {
    Configuration.PRODUCTION: 'GYQP6H83-1IC1',
    Configuration.TESTING: 'CBR8ZG83-5OS1',
    Configuration.DEVELOPMENT: 'CBR8ZG83-5OS1'
}, default=None)


def _floor_datetime_to_day(value):
    return datetime(value.year, value.month, value.day)


def _prepare_row(order):
    profit = net_profit(order)

    gclid = order.source.gclid
    conversion_name = 'zhd_import'
    conversion_time = order.finished_at.strftime(DATETIME_FORMAT)
    conversion_value = profit if profit >= Decimal(0) else Decimal(0)
    conversion_currency = 'RUB'
    return (gclid, conversion_name, conversion_time, conversion_value, conversion_currency), order.id


def net_profit(order):
    nds_deduction = Decimal('1.2')  # вычет НДС
    fee_sum = Decimal(0)

    for passenger in order.passengers:
        ticket = passenger.tickets[0]
        if ticket.places:
            fee_sum += ticket.payment.fee

    return (fee_sum / nds_deduction).quantize(Decimal('0.00'), ROUND_HALF_UP)


def _export_data_for_google_ads_task():
    log.info('Beginning export...')

    stream = BytesIO()
    csv_writer = unicodecsv.writer(stream, encoding='utf-8', dialect='excel', lineterminator='\n')
    last_processed_object_id = None
    selected_len = CHUNK_SIZE_IN_ROWS
    min_date_to_select = _floor_datetime_to_day(now_utc() - timedelta(days=DEPTH_OF_EXPORT_IN_DAYS))
    max_date_to_select = _floor_datetime_to_day(now_utc())
    processed_counter = 0

    stream.write(CSV_HEADER.encode('utf-8'))

    while selected_len == CHUNK_SIZE_IN_ROWS:
        condition = Q(source__gclid__ne=None, status=OrderStatus.DONE,
                      finished_at__gte=min_date_to_select, finished_at__lt=max_date_to_select)
        if last_processed_object_id:
            condition &= Q(id__gt=last_processed_object_id)
        orders_set = TrainOrder.objects.filter(condition).order_by('_id')[:CHUNK_SIZE_IN_ROWS]
        selected_len = len(orders_set)
        for order in orders_set:
            row, last_processed_object_id = _prepare_row(order)
            csv_writer.writerow(row)
            log.debug('TrainOrder(%s) with uid %s processed', last_processed_object_id, order.uid)
            processed_counter += 1
        log.info('%d orders processed', processed_counter)

    data = stream.getvalue()
    if data:
        mds_s3_public_client.save_data(key='google_ads/import_data.csv', data=data)
    stream.close()

    log.info('Ending export...')


def export_data_for_google_ads_task():
    try:
        try_hard(max_retries=3, sleep_duration=60)(_export_data_for_google_ads_task)()
    except Exception:
        log.exception('Error during export for Google Ads')
        dt_str = environment.now().strftime('%Y-%m-%dT%H:%M:%S')
        guaranteed_send_email(
            key='error_export_data_for_google_ads_{}'.format(dt_str),
            to_email=conf.TRAIN_PURCHASE_ERRORS_EMAIL,
            args={
                'dt': dt_str,
                'script': export_data_for_google_ads_task.__name__,
            },
            campaign=settings.ERROR_EXPORT_DATA_FOR_GOOGLE_ADS_CAMPAIGN,
        )
        raise
