# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import sys
from argparse import ArgumentParser, ArgumentTypeError, FileType
from datetime import datetime
from decimal import Decimal

import unicodecsv
from mongoengine.queryset.visitor import Q

from travel.rasp.train_api.train_purchase.core.enums import OrderStatus
from travel.rasp.train_api.train_purchase.core.models import TrainOrder

log = logging.getLogger(__name__)
CHUNK_SIZE_IN_ROWS = 500


def _export(min_date, max_date, output):
    try:
        writer = unicodecsv.writer(output, encoding='utf-8', dialect='excel', lineterminator='\n')
        headers = ['Email', 'Orders', 'Tickets', 'FeeSum']
        writer.writerow(headers)

        log.info('Reading orders...')
        total_processed = 0
        selected_len = CHUNK_SIZE_IN_ROWS
        result_dict = {}
        ids = set()

        while selected_len == CHUNK_SIZE_IN_ROWS:
            condition = Q(finished_at__gte=min_date, finished_at__lt=max_date, status=OrderStatus.DONE)
            objects_set = TrainOrder.objects.filter(condition).order_by('finished_at')[:CHUNK_SIZE_IN_ROWS]
            selected_len = 0
            for obj in objects_set:
                selected_len += 1
                if obj.id not in ids:
                    ids.add(obj.id)
                    min_date = obj.finished_at
                    mail = obj.user_info.email.lower()
                    row = result_dict.setdefault(mail, [mail, 0, 0, Decimal(0)])
                    row[1] += 1
                    row[2] += len(list(obj.iter_tickets()))
                    row[3] += sum(t.payment.fee for t in obj.iter_tickets())
                    total_processed += 1
            log.info('Processed: %d orders. %s date', total_processed, min_date)

        log.info('Writing result...')

        for email, row in result_dict.items():
            writer.writerow(row)

        log.info('Done.')
    except Exception:
        log.exception('Error in export')


def export(min_date, max_date, out_file):
    _export(_parse_date(min_date), _parse_date(max_date), FileType(mode='w')(out_file))


def _parse_date(arg):
    try:
        return datetime.strptime(arg, '%Y-%m-%d')
    except ValueError:
        raise ArgumentTypeError('{} is not valid date.'.format(arg))


def _create_parser():
    parser = ArgumentParser()
    parser.add_argument('--min_date', required=True, help='<YYYY-MM-DD>', type=_parse_date)
    parser.add_argument('--max_date', required=True, help='<YYYY-MM-DD>', type=_parse_date)
    parser.add_argument('--out_file', default=sys.stdout, help='<path/file.csv>', type=FileType(mode='w'))

    return parser


if __name__ == '__main__':
    parser = _create_parser()
    args = parser.parse_args()
    _export(min_date=args.min_date, max_date=args.max_date, output=args.out_file)
