# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import sys
from argparse import ArgumentParser, ArgumentTypeError, FileType
from datetime import datetime

import unicodecsv
from mongoengine.queryset.visitor import Q

from travel.rasp.train_api.train_purchase.core.models import TrainOrder, TrainRefund, Payment

log = logging.getLogger(__name__)
CHUNK_SIZE_IN_ROWS = 500
ORDER_COLUMNS = (
    ('OrderId', lambda o: getattr(_current_partner_data(o), 'order_num', None)),
    ('ImOrderId', lambda o: getattr(_current_partner_data(o), 'im_order_id', None)),
    ('PurchaseToken', lambda o: getattr(_current_payment(o), 'purchase_token', None)),
    ('DateTime', lambda o: o.finished_at),
    ('TrustDateTime', lambda o: getattr(_current_payment(o), 'trust_created_at', None)),
    ('ClearDateTime', lambda o: getattr(_current_payment(o), 'clear_at', None)),
    ('Amount', lambda o: sum(t.payment.amount for t in o.iter_tickets())),
    ('Insurance', lambda o: sum(i.amount for i in o.iter_insurances() if i.trust_order_id)),
    ('Fee', lambda o: sum(t.payment.fee for t in o.iter_tickets())),
)
REFUND_COLUMNS = (
    ('OrderId', lambda r: getattr(_current_partner_data(r.order), 'order_num', None)),
    ('ImOrderId', lambda r: getattr(_current_partner_data(r.order), 'im_order_id', None)),
    ('PurchaseToken', lambda r: getattr(r.refund_payment, 'purchase_token', None)),
    ('TrustRefundId', lambda r: getattr(r.refund_payment, 'trust_refund_id', None)),
    ('TrustReversalId', lambda r: getattr(r.refund_payment, 'trust_reversal_id', None)),
    ('DateTime', lambda r: r.created_at),
    ('TrustDateTime', lambda r: getattr(r.refund_payment, 'refund_created_at', None)),
    ('Amount', lambda r: sum(t.payment.amount for t in r.order.iter_tickets() if t.blank_id in r.blank_ids)),
    ('Insurance', lambda r: sum(p.insurance.amount for p in r.order.passengers
                                if (getattr(p.insurance, 'operation_id', None) in r.insurance_ids)
                                and getattr(p.insurance, 'trust_order_id', None))),
)


def _current_partner_data(order):
    return order.partner_data_history[-1] if order.partner_data_history else None


def _current_payment(order):
    return Payment.objects.filter(order_uid=order.uid).order_by('-trust_created_at').first()


def _export(writer, source, condition, columns):
    last_processed_object_id = None
    total_processed = 0
    selected_len = CHUNK_SIZE_IN_ROWS
    headers = [col[0] for col in columns]
    writer.writerow(headers)

    while selected_len == CHUNK_SIZE_IN_ROWS:
        if last_processed_object_id:
            condition &= Q(id__gt=last_processed_object_id)
        objects_set = source.objects.filter(condition).order_by('_id')[:CHUNK_SIZE_IN_ROWS]
        selected_len = len(objects_set)
        total_processed += selected_len
        for obj in objects_set:
            last_processed_object_id = obj.id
            values = [col[1](obj) for col in columns]
            writer.writerow(values)
        log.debug('Total processed: %d rows.', total_processed)


def export_orders_and_refunds(min_date, max_date, stream_for_orders, stream_for_refunds):
    try:
        writer_for_orders = unicodecsv.writer(stream_for_orders, encoding='utf-8', dialect='excel', lineterminator='\n')
        writer_for_refunds = unicodecsv.writer(stream_for_refunds, encoding='utf-8', dialect='excel',
                                               lineterminator='\n')

        log.info('Exporting orders...')
        condition = Q(finished_at__gte=min_date, finished_at__lt=max_date)
        _export(writer=writer_for_orders, source=TrainOrder, condition=condition, columns=ORDER_COLUMNS)

        log.info('Exporting refunds...')
        condition = Q(created_at__gte=min_date, created_at__lt=max_date)
        _export(writer=writer_for_refunds, source=TrainRefund, condition=condition, columns=REFUND_COLUMNS)

        log.info('Done.')

    except Exception:
        log.exception('Error in export_orders_and_refunds')


def _create_parser():
    def _parse_date(arg):
        try:
            return datetime.strptime(arg, '%Y-%m-%d')
        except ValueError:
            raise ArgumentTypeError('{} is not valid date.'.format(arg))

    parser = ArgumentParser()
    parser.add_argument('--min_date', required=True, help='<YYYY-MM-DD>', type=_parse_date)
    parser.add_argument('--max_date', required=True, help='<YYYY-MM-DD>', type=_parse_date)
    parser.add_argument('--orders_out', default=sys.stdout, help='<path/file.csv>', type=FileType(mode='w'))
    parser.add_argument('--refunds_out', default=sys.stdout, help='<path/file.csv>', type=FileType(mode='w'))

    return parser


def main():
    parser = _create_parser()
    args = parser.parse_args()
    export_orders_and_refunds(min_date=args.min_date, max_date=args.max_date, stream_for_orders=args.orders_out,
                              stream_for_refunds=args.refunds_out)
