# coding: utf8
"""
    скрипт для возврата средств по заказам
    данные в монге не обновляются! только правит поле admin_comment при успешном возврате

    >manage.py shell

    >from travel.rasp.train_api.scripts.force_payment_refund import *
    >uids = ['','','']  # тут список заказов к возврату
    >  # далее возможны варианты:
    >try_many(uids, refund_full)  # для полного возврата средств по заказу
    >try_many(uids, refund_fee)  # для возврата только нашего сбора

    для возврата по единичному заказу можно юзать refund_full(uid) или refund_fee(uid)

    внимательно смотрим вывод консоли!
"""
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from decimal import Decimal

from ylog.context import log_context

from common.data_api.billing.trust_client import (
    TrustClient, TrustClientRequestError, TrustRefundOrder, TrustRefundStatuses
)
from travel.rasp.library.python.common23.date.environment import now_utc
from common.utils.exceptions import SimpleUnicodeException
from common.utils.try_hard import try_hard
from travel.rasp.train_api.train_purchase.core.models import (
    TrainOrder, TrainRefund, RefundPayment, RefundPaymentStatus
)
from travel.rasp.train_api.train_purchase.workflow.payment.refund_payment import (
    start_refund_payment, WaitingForTrustRefundStatus
)

log = logging.getLogger(__name__)


def refund_custom(order_uid,
                  service_refund_amount=Decimal("0.00"),
                  ticket_refund_amount=Decimal("0.00"),
                  fee_refund_amount=Decimal("0.00")):
    with log_context(order_uid=order_uid):
        total_refund_amount = service_refund_amount + ticket_refund_amount + fee_refund_amount
        order = TrainOrder.objects.get(uid=order_uid)
        trust_client = TrustClient(user_ip=order.user_info.ip, user_region_id=order.user_info.region_id)
        refund_orders = []

        for ticket in order.iter_tickets():
            if service_refund_amount:
                delta_amount = Decimal.min(service_refund_amount, ticket.payment.service_amount)
                service_refund_amount -= delta_amount
                refund_orders.append(TrustRefundOrder(order_id=ticket.payment.service_order_id,
                                                      delta_amount=delta_amount))
            if ticket_refund_amount:
                delta_amount = Decimal.min(ticket_refund_amount, ticket.payment.ticket_amount)
                ticket_refund_amount -= delta_amount
                refund_orders.append(TrustRefundOrder(order_id=ticket.payment.ticket_order_id,
                                                      delta_amount=delta_amount))
            if fee_refund_amount:
                delta_amount = Decimal.min(fee_refund_amount, ticket.payment.fee)
                fee_refund_amount -= delta_amount
                refund_orders.append(TrustRefundOrder(order_id=ticket.payment.fee_order_id,
                                                      delta_amount=delta_amount))

        if service_refund_amount or ticket_refund_amount or fee_refund_amount:
            raise SimpleUnicodeException('Невозможно вернуть указанную сумму ({}). uid={} purchase_token={}'.format(
                total_refund_amount,
                order_uid,
                order.current_billing_payment.purchase_token
            ))

        if refund_orders:
            trust_refund_id = trust_client.create_refund(refund_orders, order.current_billing_payment.purchase_token)
        else:
            log.info('Nothing to return. uid={} purchase_token={}'.format(
                order_uid,
                order.current_billing_payment.purchase_token
            ))
            return

        start_refund_payment(trust_client, trust_refund_id)
        log.info('Refund created. uid={} purchase_token={} trust_refund_id={}'.format(
            order_uid,
            order.current_billing_payment.purchase_token,
            trust_refund_id
        ))
        check_refund_payment(trust_client, trust_refund_id)
        refund_check = trust_client.get_refund_receipt_url(trust_refund_id)
        log.info('Refund completed. uid={} purchase_token={} trust_refund_id={}. Refund check {}'.format(
            order_uid,
            order.current_billing_payment.purchase_token,
            trust_refund_id,
            refund_check
        ))
        order.update(set__admin_comment='Возвращена указанная сумма ({}) trust_refund_id={}'
                     .format(total_refund_amount, trust_refund_id))


def refund_full_current_amount(order_uid):
    with log_context(order_uid=order_uid):
        order = TrainOrder.objects.get(uid=order_uid)
        trust_client = TrustClient(user_ip=order.user_info.ip, user_region_id=order.user_info.region_id)
        refund_orders = []

        payment_info = trust_client.get_raw_payment_info(order.current_billing_payment.purchase_token)
        for trust_order in payment_info['orders']:
            paid_amount = Decimal(trust_order['paid_amount'])
            if not paid_amount.is_zero():
                refund_orders.append(TrustRefundOrder(order_id=trust_order['order_id'], delta_amount=paid_amount))

        if refund_orders:
            trust_refund_id = trust_client.create_refund(refund_orders, order.current_billing_payment.purchase_token)
        else:
            log.info('Nothing to return. uid={} purchase_token={}'.format(
                order_uid,
                order.current_billing_payment.purchase_token
            ))
            return

        start_refund_payment(trust_client, trust_refund_id)
        log.info('Refund created. uid={} purchase_token={} trust_refund_id={}'.format(
            order_uid,
            order.current_billing_payment.purchase_token,
            trust_refund_id
        ))
        check_refund_payment(trust_client, trust_refund_id)
        refund_check = trust_client.get_refund_receipt_url(trust_refund_id)
        log.info('Refund completed. uid={} purchase_token={} trust_refund_id={}. Refund check {}'.format(
            order_uid,
            order.current_billing_payment.purchase_token,
            trust_refund_id,
            refund_check
        ))
        order.update(set__admin_comment='Возвращена полная стоимость по состоянию из траста trust_refund_id={}'
                     .format(trust_refund_id))


def refund_full(order_uid):
    with log_context(order_uid=order_uid):
        order = TrainOrder.objects.get(uid=order_uid)
        trust_client = TrustClient(user_ip=order.user_info.ip, user_region_id=order.user_info.region_id)
        trust_refund_id = create_refund_payment(trust_client, order, return_ticket=True, return_fee=True)
        start_refund_payment(trust_client, trust_refund_id)
        log.info('Refund created. uid={} purchase_token={} trust_refund_id={}'.format(
            order_uid,
            order.current_billing_payment.purchase_token,
            trust_refund_id
        ))
        check_refund_payment(trust_client, trust_refund_id)
        refund_check = trust_client.get_refund_receipt_url(trust_refund_id)
        log.info('Refund completed. uid={} purchase_token={} trust_refund_id={}. Refund check {}'.format(
            order_uid,
            order.current_billing_payment.purchase_token,
            trust_refund_id,
            refund_check
        ))
        order.update(set__admin_comment='Возвращена полная стоимость trust_refund_id={}'.format(trust_refund_id))


def refund_fee(order_uid):
    with log_context(order_uid=order_uid):
        order = TrainOrder.objects.get(uid=order_uid)
        trust_client = TrustClient(user_ip=order.user_info.ip, user_region_id=order.user_info.region_id)
        trust_refund_id = create_refund_payment(trust_client, order, return_ticket=False, return_fee=True)
        start_refund_payment(trust_client, trust_refund_id)
        log.info('Refund created. uid={} purchase_token={} trust_refund_id={}'.format(
            order_uid,
            order.current_billing_payment.purchase_token,
            trust_refund_id
        ))
        check_refund_payment(trust_client, trust_refund_id)
        refund_check = trust_client.get_refund_receipt_url(trust_refund_id)
        log.info('Refund completed. uid={} purchase_token={} trust_refund_id={}. Refund check {}'.format(
            order_uid,
            order.current_billing_payment.purchase_token,
            trust_refund_id,
            refund_check
        ))
        order.update(set__admin_comment='Возвращен сбор trust_refund_id={}'.format(trust_refund_id))


@try_hard(retriable_exceptions=(TrustClientRequestError,))
def create_refund_payment(trust_client, order, return_ticket=True, return_fee=False, return_insurance=False,
                          refund_blank_ids=[]):
    refund_orders = []

    for passenger in order.passengers:
        ticket = passenger.tickets[0]

        if refund_blank_ids and ticket.blank_id not in refund_blank_ids:
            continue

        if return_ticket:
            service_refund_amount = ticket.payment.service_amount
            ticket_refund_amount = ticket.payment.amount

            if service_refund_amount:
                if ticket_refund_amount < service_refund_amount:
                    log.info('Unable to return the full cost of services. To return: {0}; services: {1}.'
                             .format(ticket_refund_amount, service_refund_amount))
                    service_refund_amount = ticket_refund_amount
                ticket_refund_amount -= service_refund_amount
                refund_orders.append(TrustRefundOrder(order_id=ticket.payment.service_order_id,
                                                      delta_amount=service_refund_amount))
            if ticket_refund_amount:
                refund_orders.append(TrustRefundOrder(order_id=ticket.payment.ticket_order_id,
                                                      delta_amount=ticket_refund_amount))

        if return_fee:
            fee_refund_amount = ticket.payment.fee

            if fee_refund_amount:
                refund_orders.append(TrustRefundOrder(order_id=ticket.payment.fee_order_id,
                                                      delta_amount=fee_refund_amount))

        if return_insurance:
            insurance = passenger.insurance
            if insurance and insurance.trust_order_id:
                refund_orders.append(TrustRefundOrder(order_id=insurance.trust_order_id,
                                                      delta_amount=insurance.amount))

    if refund_orders:
        return trust_client.create_refund(refund_orders, order.current_billing_payment.purchase_token)

    return None


def create_pending_refund_payment(refund_uuid):
    refund = TrainRefund.objects.get(uuid=refund_uuid)
    payment = refund.order.current_billing_payment
    paid_insurance_ids = set(p.insurance.operation_id for p in refund.order.passengers
                             if p.insurance and p.insurance.operation_id and p.insurance.trust_order_id)
    refund_insurance_ids = list(set(refund.insurance_ids) & paid_insurance_ids)
    refund_payment = RefundPayment.objects.create(
        order_uid=refund.order_uid,
        purchase_token=payment.purchase_token,
        refund_uuid=refund.uuid,
        refund_payment_status=RefundPaymentStatus.NEW,
        refund_created_at=now_utc(),
        refund_blank_ids=refund.blank_ids,
        refund_insurance_ids=refund_insurance_ids,
        user_info=refund.user_info,
    )
    return refund_payment


@try_hard(max_retries=10, sleep_duration=6,
          retriable_exceptions=(TrustClientRequestError, WaitingForTrustRefundStatus))
def check_refund_payment(trust_client, trust_refund_id):
    if trust_client.get_refund_status(trust_refund_id) != TrustRefundStatuses.SUCCESS:
        raise WaitingForTrustRefundStatus('Статус возврата в биллинге еще не SUCCESS.')


def try_many(uids, method):
    for uid in uids:
        try:
            method(uid)
            log.info('Done {} with uid={}'.format(method.__name__, uid))
        except Exception:
            log.exception('Failed {} with uid={}'.format(method.__name__, uid))
