# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import sys
from datetime import datetime, time

from dateutil.relativedelta import relativedelta

from travel.rasp.library.python.common23.date import environment
from common.utils.date import MSK_TZ, UTC_TZ
from common.utils.unicode_csv import UnicodeDictWriter
from travel.rasp.train_api.train_purchase.core.models import TrainOrder

REPORT_FIELDS = (
    'order_uid',
    'purchase_token',
    'order_num',
    'created_at',
    'finished_at',
    'tickets_count',
    'tickets_sum',
    'fees_sum',
    'partner_fees_sum',
    'partner_code'
)


def as_msk(utc_dt):
    if utc_dt.tzinfo is None:
        utc_dt = UTC_TZ.localize(utc_dt)
    return utc_dt.astimezone(MSK_TZ)


def write_orders_report(stream, start_dt, end_dt):
    writer = UnicodeDictWriter(stream, fieldnames=REPORT_FIELDS, dialect='excel-tab', encoding='utf-8')
    writer.writehead()
    for order in TrainOrder.objects.filter(finished_at__gte=start_dt, finished_at__lt=end_dt):
        tickets = tuple(order.iter_tickets())
        order_payments = [ticket.payment for ticket in tickets]
        writer.writerow({
            'order_uid': order.uid,
            'purchase_token': order.current_billing_payment.purchase_token,
            'order_num': order.current_partner_data.order_num,
            'created_at': as_msk(order.id.generation_time),
            'finished_at': as_msk(order.finished_at),
            'tickets_count': len(tickets),
            'tickets_sum': sum(payment.amount for payment in order_payments),
            'fees_sum': sum(payment.fee for payment in order_payments),
            'partner_fees_sum': sum(payment.partner_fee for payment in order_payments),
            'partner_code': order.partner
        })


def orders_report(filename='-', month=None):
    report_file = open(filename, 'w') if filename != '-' else sys.stdout

    month_start = (datetime.strptime(month, '%Y-%m') if month is not None else
                   datetime.combine(environment.today().replace(day=1), time.min))
    month_end = month_start + relativedelta(months=1)

    write_orders_report(report_file, MSK_TZ.localize(month_start), MSK_TZ.localize(month_end))
