# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging

from MySQLdb.cursors import OperationalError

from common.celery.task import single_launch_task
from common.models.geo import Station
from common.utils.request_limiter import RequestLimiter
from travel.rasp.train_api.train_partners.base.get_route_info import get_route_info_for_order
from travel.rasp.train_api.train_purchase.core.enums import TrainPartner
from travel.rasp.train_api.train_purchase.core.models import TrainOrder, OrderRouteInfo, StationInfo

log = logging.getLogger(__name__)
request_limiter = RequestLimiter(max_rps=5)
CHUNK_SIZE_IN_ROWS = 500


@request_limiter.throttle
def _set_route_info_for_order(order):
    try:
        log.debug('Processing order %s', order.uid)
        TrainOrder.fetch_stations([order])
        order.route_info = OrderRouteInfo(
            from_station=StationInfo.create_from_station(order.station_from),
            to_station=StationInfo.create_from_station(order.station_to),
        )
        result = get_route_info_for_order(order) if order.partner == TrainPartner.IM else None
        if result:
            try:
                start_station = Station.get_by_code('express', result.first_stop.station_express_code)
                order.route_info.start_station = StationInfo.create_from_station(start_station)
                order.route_info.start_station.departure = result.first_stop.departure_datetime
            except Station.DoesNotExist:
                log.warning('Cannot get station by code %s', result.first_stop.station_express_code)

            try:
                end_station = Station.get_by_code('express', result.last_stop.station_express_code)
                order.route_info.end_station = StationInfo.create_from_station(end_station)
            except Station.DoesNotExist:
                log.warning('Cannot get station by code %s', result.last_stop.station_express_code)

            if order.partner_data_history:
                order.current_partner_data.start_station_title = result.first_stop.station_name
                order.current_partner_data.end_station_title = result.last_stop.station_name
        order.save()
    except OperationalError:
        log.warning('Error during _set_route_info_for_order', exc_info=1)


@single_launch_task()
def set_route_info_task():
    try:
        log.info('Starting migration set_route_info_task')
        selected_len = CHUNK_SIZE_IN_ROWS
        processed_count = 0

        while selected_len == CHUNK_SIZE_IN_ROWS:
            orders = TrainOrder.objects.filter(uid__ne=None, route_info=None).order_by('_id')[:CHUNK_SIZE_IN_ROWS]
            selected_len = len(orders)
            for order in orders:
                _set_route_info_for_order(order)
            processed_count += selected_len
            log.info('%d orders processed.', processed_count)

        log.info('Ending migration set_route_info_task')
        return 0
    except Exception:
        log.critical('Error during set_route_info_task', exc_info=1)
        return -1


if __name__ == '__main__':
    set_route_info_task()
