# -*- encoding: utf-8 -*-
from __future__ import unicode_literals, absolute_import, division, print_function

"""
    https://github.yandex-team.ru/avia/travelers/blob/release/application/services/data_sync/client.py
"""
from datetime import datetime
from typing import List, Optional  # noqa

import requests
from django.conf import settings

from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from common.data_api.tvm.instance import tvm_factory
from travel.rasp.train_api.scripts.sync_travelers import schemas
from travel.rasp.train_api.scripts.sync_travelers.models import Traveler, Passenger, Document, BonusCard  # noqa

define_setting('DATA_SYNC_API_URL', {
    Configuration.PRODUCTION: 'https://datasync.yandex.net:8443/v1/personality/profile',
    Configuration.STRESS: 'http://dataapi-load01g.dst.yandex.net:21859/v1/personality/profile',
}, default='http://dataapi.dst.yandex.net:21859/v1/personality/profile')

define_setting('DATA_SYNC_TIMEOUT', default=5)


class DataSyncError(RuntimeError):
    pass


class ForbiddenDataSyncError(DataSyncError):
    pass


class DataSyncClient(object):
    def __init__(self, api_url, tvm_id, tvm_client, timeout):
        self._api_url = api_url
        self._tvm_id = tvm_id
        self._tvm_client = tvm_client
        self._timeout = timeout
        self._traveler_schema = schemas.TravelerSchema()
        self._passenger_schema = schemas.PassengerSchema()
        self._passengers_schema = schemas.PassengerSchema(many=True)
        self._document_schema = schemas.DocumentSchema()
        self._documents_schema = schemas.DocumentSchema(many=True)
        self._bonus_card_schema = schemas.BonusCardSchema()
        self._bonus_cards_schema = schemas.BonusCardSchema(many=True)

    def get_traveler(self, uid, user_ticket=None):
        # type: (str, str) -> Traveler
        response = self._get('/traveler', uid=uid, user_ticket=user_ticket)
        items = response.get('items', [])

        return self._traveler_schema.load(items[0]) if len(items) > 0 else None

    def save_traveler(self, uid, traveler, user_ticket=None):
        # type: (str, Traveler, str) -> Traveler
        traveler.updated_at = datetime.now()
        if traveler.id:
            response = self._put(
                '/traveler/{}'.format(traveler.id),
                data=self._traveler_schema.dump(traveler),
                uid=uid,
                user_ticket=user_ticket,
            )
        else:
            traveler.created_at = datetime.now()
            response = self._post(
                '/traveler',
                data=self._traveler_schema.dump(traveler),
                uid=uid,
                user_ticket=user_ticket,
            )

        return self._traveler_schema.load(response)

    def get_passenger(self, uid, passenger_id, user_ticket=None):
        # type: (str, str, str) -> Optional[Passenger]
        passengers = self.get_passengers(uid, user_ticket)
        for passenger in passengers:
            if str(passenger.id) == passenger_id:
                return passenger

        return None

    def get_passengers(self, uid, user_ticket=None):
        # type: (str, str) -> List[Passenger]
        response = self._get(
            '/traveler/passengers',
            uid=uid,
            user_ticket=user_ticket,
        )

        return self._passengers_schema.load(response.get('items', []))

    def save_passenger(self, uid, passenger, user_ticket=None):
        # type: (str, Passenger, str) -> Passenger
        passenger.updated_at = datetime.now()

        if passenger.id:
            response = self._put(
                '/traveler/passengers/' + str(passenger.id),
                data=self._passenger_schema.dump(passenger),
                uid=uid,
                user_ticket=user_ticket,
            )
        else:
            passenger.created_at = datetime.now()
            response = self._post(
                '/traveler/passengers',
                data=self._passenger_schema.dump(passenger),
                uid=uid,
                user_ticket=user_ticket,
            )

        return self._passenger_schema.load(response)

    def delete_passenger(self, uid, passenger, user_ticket=None):
        # type: (str, Passenger, str) -> bool
        documents = self.get_documents(uid, user_ticket, passenger)
        for document in documents:
            self.delete_document(uid, user_ticket, document)

        self._delete(
            '/traveler/passengers/' + str(passenger.id),
            uid=uid,
            user_ticket=user_ticket,
        )

        return True

    def get_document(self, uid, passenger, document_id, user_ticket=None):
        # type: (str, str, Passenger, str) -> Optional[Document]
        documents = self.get_documents(uid, user_ticket, passenger)
        for document in documents:
            if str(document.id) == document_id:
                return document

        return None

    def get_documents(self, uid, passenger=None, user_ticket=None):
        # type: (str, Passenger, str) -> List[Document]
        response = self._get(
            '/traveler/passengers/documents',
            uid=uid,
            user_ticket=user_ticket,
        )

        all_documents = self._documents_schema.load(response.get('items', []))
        if passenger is None:
            return all_documents
        else:
            return [d for d in all_documents if d.passenger_id == passenger.id]

    def save_document(self, uid, passenger, document, user_ticket=None):
        # type: (str, Passenger, Document, str) -> Document
        document.passenger_id = passenger.id
        document.updated_at = datetime.now()

        if document.id:
            response = self._put(
                '/traveler/passengers/documents/' + str(document.id),
                data=self._document_schema.dump(document),
                uid=uid,
                user_ticket=user_ticket,
            )
        else:
            document.created_at = datetime.now()
            response = self._post(
                '/traveler/passengers/documents',
                data=self._document_schema.dump(document),
                uid=uid,
                user_ticket=user_ticket,
            )

        return self._document_schema.load(response)

    def delete_document(self, uid, document, user_ticket=None):
        # type: (str, str, Document) -> bool
        self._delete(
            '/traveler/passengers/documents/' + str(document.id),
            uid=uid,
            user_ticket=user_ticket,
        )

        return True

    def get_bonus_card(self, uid, passenger, bonus_card_id, user_ticket=None):
        # type: (str, str, Passenger, str) -> Optional[BonusCard]
        bonus_cards = self.get_bonus_cards(uid, user_ticket, passenger)
        for bonus_card in bonus_cards:
            if str(bonus_card.id) == bonus_card_id:
                return bonus_card

        return None

    def get_bonus_cards(self, uid, passenger, user_ticket=None):
        # type: (str, Passenger, str) -> List[BonusCard]
        response = self._get(
            '/traveler/passengers/bonus_cards',
            uid=uid,
            user_ticket=user_ticket,
        )

        result = []
        for d in self._bonus_cards_schema.load(response.get('items', [])):
            if d.passenger_id == passenger.id:
                result.append(d)

        return result

    def save_bonus_card(self, uid, passenger, bonus_card, user_ticket=None):
        # type: (str, Passenger, BonusCard, str) -> BonusCard
        bonus_card.passenger_id = passenger.id
        bonus_card.updated_at = datetime.now()

        if bonus_card.id:
            response = self._put(
                '/traveler/passengers/bonus_cards/' + str(bonus_card.id),
                data=self._bonus_card_schema.dump(bonus_card),
                uid=uid,
                user_ticket=user_ticket,
            )
        else:
            bonus_card.created_at = datetime.now()
            response = self._post(
                '/traveler/passengers/bonus_cards',
                data=self._bonus_card_schema.dump(bonus_card),
                uid=uid,
                user_ticket=user_ticket,
            )

        return self._bonus_card_schema.load(response)

    def delete_bonus_card(self, uid, bonus_card, user_ticket=None):
        # type: (str, BonusCard, str) -> bool
        self._delete(
            '/traveler/passengers/bonus_cards/' + str(bonus_card.id),
            uid=uid,
            user_ticket=user_ticket,
        )

        return True

    def _get(self, path, uid, user_ticket):
        service_ticket = self._tvm_client.get_ticket(self._tvm_id)
        headers = {'X-Uid': uid, 'X-Ya-Service-Ticket': service_ticket}
        if user_ticket:
            headers['X-Ya-User-Ticket'] = user_ticket
        response = requests.get(
            self._api_url + path,
            headers=headers,
            verify=False,
            timeout=self._timeout,
        )

        if response.status_code == 404:
            return None

        self._raise_for_status(response)

        return response.json()

    def _post(self, path, data, uid, user_ticket):
        service_ticket = self._tvm_client.get_ticket(self._tvm_id)
        headers = {
            'X-Uid': uid,
            'X-Ya-Service-Ticket': service_ticket,
            'Content-Type': 'application/json',
        }
        if user_ticket:
            headers['X-Ya-User-Ticket'] = user_ticket
        response = requests.post(
            self._api_url + path,
            headers=headers,
            json=data,
            verify=False,
            timeout=self._timeout,
        )

        self._raise_for_status(response)

        return response.json()

    def _put(self, path, data, uid, user_ticket):
        service_ticket = self._tvm_client.get_ticket(self._tvm_id)
        headers = {'X-Uid': uid, 'X-Ya-Service-Ticket': service_ticket}
        if user_ticket:
            headers['X-Ya-User-Ticket'] = user_ticket
        response = requests.put(
            self._api_url + path,
            headers=headers,
            json=data,
            verify=False,
            timeout=self._timeout,
        )

        self._raise_for_status(response)

        return response.json()

    def _delete(self, path, uid, user_ticket):
        service_ticket = self._tvm_client.get_ticket(self._tvm_id)
        headers = {'X-Uid': uid, 'X-Ya-Service-Ticket': service_ticket}
        if user_ticket:
            headers['X-Ya-User-Ticket'] = user_ticket
        response = requests.delete(
            self._api_url + path,
            headers=headers,
            verify=False,
            timeout=self._timeout,
        )

        self._raise_for_status(response)

        return response.json()

    @staticmethod
    def _raise_for_status(response):
        # type: (requests.Response) -> None
        try:
            response.raise_for_status()
        except requests.HTTPError:
            if response.status_code == 403:
                raise ForbiddenDataSyncError()

            raise DataSyncError('%s DataSync error: %s for %s\nResponse:\n%s' % (
                response.status_code,
                response.reason,
                response.url,
                response.content,
            ))


data_sync_client = DataSyncClient(
    settings.DATA_SYNC_API_URL,
    settings.TVM_DATASYNC,
    tvm_factory.get_provider(),
    settings.DATA_SYNC_TIMEOUT,
)
