# -*- encoding: utf-8 -*-
from __future__ import unicode_literals, absolute_import, division, print_function

"""
    https://github.yandex-team.ru/avia/travelers/blob/release/application/services/data_sync/schemas.py
"""
from enum import Enum
from marshmallow import Schema, fields, post_load, post_dump
from marshmallow_enum import EnumField

from travel.rasp.train_api.scripts.sync_travelers.models import Traveler, Passenger, Document, BonusCard


class Gender(Enum):
    male = 'male'
    female = 'female'


class DocumentType(Enum):
    ru_national_passport = 'ru_national_passport'
    ru_foreign_passport = 'ru_foreign_passport'
    ru_birth_certificate = 'ru_birth_certificate'
    ru_seaman_passport = 'ru_seaman_passport'
    ru_military_id = 'ru_military_id'
    other = 'other'


class ShemaError(Exception):
    def __init__(self, shema_type_name, action, errors):
        self.errors = errors
        self.shema_type_name = shema_type_name
        self.action = action

    def __str__(self):
        return u'Failed {}.{}(). Errors:{}'.format(self.shema_type_name, self.action, u'\r\n'.join(self.errors))

    def __repr__(self):
        return self.__str__()


class BaseDataSyncSchema(Schema):
    @post_dump
    def null_to_str(self, data):
        result = dict()
        for k in data:
            if data[k] is not None:
                result[k] = data[k]

        return result

    def load(self, *args, **kwargs):
        result = super(BaseDataSyncSchema, self).load(*args, **kwargs)
        if result.errors:
            raise ShemaError(type(self).__name__, u'load', result.errors)
        return result.data

    def dump(self, *args, **kwargs):
        result = super(BaseDataSyncSchema, self).dump(*args, **kwargs)
        if result.errors:
            raise ShemaError(type(self).__name__, u'dump', result.errors)
        return result.data


class TravelerSchema(BaseDataSyncSchema):
    id = fields.UUID(load_only=True, data_key='id', attribute='id')
    agree = fields.Boolean()
    phone = fields.String()
    phone_additional = fields.String()
    email = fields.String()
    created_at = fields.DateTime(format='%Y-%m-%dT%H:%M:%S.000Z')
    updated_at = fields.DateTime(format='%Y-%m-%dT%H:%M:%S.000Z')

    @post_load
    def make_traveler(self, data):
        return Traveler().fill(data)


class PassengerSchema(BaseDataSyncSchema):
    id = fields.UUID(load_only=True, data_key='id', attribute='id')
    traveler_id = fields.UUID()
    title = fields.String()
    gender = EnumField(Gender)
    birth_date = fields.DateTime(format='%Y-%m-%dT%H:%M:%S.000Z')
    phone = fields.String()
    phone_additional = fields.String()
    email = fields.String()
    created_at = fields.DateTime(format='%Y-%m-%dT%H:%M:%S.000Z')
    updated_at = fields.DateTime(format='%Y-%m-%dT%H:%M:%S.000Z')

    @post_load
    def make_passenger(self, data):
        return Passenger().fill(data)


class DocumentSchema(BaseDataSyncSchema):
    id = fields.UUID(load_only=True, data_key='id', attribute='id')
    passenger_id = fields.UUID()
    type = EnumField(DocumentType)
    title = fields.String()
    number = fields.String()
    first_name = fields.String()
    middle_name = fields.String()
    last_name = fields.String()
    first_name_en = fields.String()
    middle_name_en = fields.String()
    last_name_en = fields.String()
    issue_date = fields.DateTime(format='%Y-%m-%dT%H:%M:%S.000Z')
    expiration_date = fields.DateTime(format='%Y-%m-%dT%H:%M:%S.000Z')
    citizenship = fields.String()
    created_at = fields.DateTime(format='%Y-%m-%dT%H:%M:%S.000Z')
    updated_at = fields.DateTime(format='%Y-%m-%dT%H:%M:%S.000Z')

    @post_load
    def make_document(self, data):
        return Document().fill(data)


class BonusCardSchema(BaseDataSyncSchema):
    id = fields.UUID(load_only=True, data_key='id', attribute='id')
    passenger_id = fields.UUID()
    type = fields.String()
    title = fields.String()
    number = fields.String()
    created_at = fields.DateTime(format='%Y-%m-%dT%H:%M:%S.000Z')
    updated_at = fields.DateTime(format='%Y-%m-%dT%H:%M:%S.000Z')

    @post_load
    def make_bonus_card(self, data):
        return BonusCard().fill(data)
