# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from enum import IntEnum

from common.utils.exceptions import SimpleUnicodeException
from common.utils.request_limiter import RequestLimiter
from common.utils.try_hard import try_hard
from travel.rasp.train_api.scripts.sync_travelers.client import data_sync_client
from travel.rasp.train_api.scripts.sync_travelers.models import Traveler, Passenger, Document
from travel.rasp.train_api.scripts.sync_travelers.schemas import Gender as DsGender, DocumentType as DsDocumentType
from travel.rasp.train_api.train_partners.base.get_order_info import get_order_info as get_order_info_original
from travel.rasp.train_api.train_purchase.core.enums import OrderStatus, Gender, DocumentType
from travel.rasp.train_api.train_purchase.core.models import TrainOrder

log = logging.getLogger(__name__)
request_limiter = RequestLimiter(max_rps=2)

DATASYNC_DOC_TYPE_MAP = {
    DocumentType.RUSSIAN_PASSPORT: DsDocumentType.ru_national_passport,
    DocumentType.BIRTH_CERTIFICATE: DsDocumentType.ru_birth_certificate,
    DocumentType.FOREIGN_DOCUMENT: DsDocumentType.other,
    DocumentType.MILITARY_CARD: DsDocumentType.ru_military_id,
    DocumentType.RUSSIAN_INTERNATIONAL_PASSPORT: DsDocumentType.ru_foreign_passport,
    DocumentType.SAILOR_PASSPORT: DsDocumentType.ru_seaman_passport,
}

DATASYNC_DOC_TITLE_MAP = {
    DocumentType.RUSSIAN_PASSPORT: 'Паспорт РФ',
    DocumentType.BIRTH_CERTIFICATE: 'Свидетельство о рождении',
    DocumentType.FOREIGN_DOCUMENT: 'Иностранный документ',
    DocumentType.MILITARY_CARD: 'Военный билет',
    DocumentType.RUSSIAN_INTERNATIONAL_PASSPORT: 'Заграничный паспорт',
    DocumentType.SAILOR_PASSPORT: 'Паспорт моряка',
}


class SyncStatus(IntEnum):
    STARTED = 1
    DONE = 2
    FAILED = 3


class PassengerGroup(object):
    def __init__(self, default_order, default_order_passenger):
        self.default_order = default_order
        self.passenger = default_order_passenger
        self.orders = []
        self.passenger_info = None

    def get_key(self):
        return (self.passenger.first_name, self.passenger.patronymic, self.passenger.last_name,
                self.passenger.sex)

    def set_order_info(self, order_info):
        self.passenger_info = None
        for pi in order_info.passengers:
            if self.passenger.tickets[0].blank_id == pi.blank_id:
                self.passenger_info = pi
        if not self.passenger_info:
            raise SimpleUnicodeException('Не нашли blank_id={} для заказа {}'.format(
                self.passenger.tickets[0].blank_id, self.default_order.uid
            ))

    def create_datasync_passenger(self, traveler_id=None):
        ds_passenger = Passenger()
        ds_passenger.traveler_id = traveler_id
        ds_passenger.gender = DsGender.female if self.passenger.sex == Gender.FEMALE else DsGender.male
        ds_passenger.birth_date = self.passenger_info.birth_date
        title_format = '{0} {1} {2}' if self.passenger.patronymic and self.passenger.patronymic != '-' else '{0} {2}'
        ds_passenger.title = title_format.format(
            self.passenger.first_name, self.passenger.patronymic, self.passenger.last_name
        )
        return ds_passenger

    def create_datasync_document(self, passenger_id=None):
        ds_doc = Document()
        ds_doc.passenger_id = passenger_id
        ds_doc.first_name = self.passenger.first_name
        ds_doc.middle_name = (
            self.passenger.patronymic if self.passenger.patronymic and self.passenger.patronymic != '-' else ''
        )
        ds_doc.last_name = self.passenger.last_name
        ds_doc.citizenship = str(self.passenger.citizenship_country._geo_id)
        ds_doc.type = DATASYNC_DOC_TYPE_MAP[self.passenger.doc_type]
        ds_doc.title = DATASYNC_DOC_TITLE_MAP[self.passenger.doc_type]
        ds_doc.number = self.passenger_info.doc_id
        return ds_doc


def get_orders_by_user(user_uid):
    return TrainOrder.objects.filter(
        user_info__uid=user_uid,
        status=OrderStatus.DONE,
    ).order_by('-_id')


def get_user_uids_to_sync():
    query = TrainOrder.objects.aggregate(*[
        {
            '$match': {
                'user_info.uid': {'$exists': True},
                'status': OrderStatus.DONE.value,
                'sync_personal_data_status': {'$exists': False},
            }
        },
        {
            '$group': {
                '_id': '$user_info.uid',
            }
        },
    ])
    return [row['_id'] for row in query]


def sync_user_orders(user_uid, orders):
    try:
        log.debug('user={}: started'.format(user_uid))
        ds_traveler = data_sync_client.get_traveler(user_uid)
        if not ds_traveler:
            order = orders[0]
            ds_traveler = Traveler()
            ds_traveler.phone = order.user_info.phone
            ds_traveler.email = order.user_info.email
            ds_traveler.agree = True
            ds_traveler = data_sync_client.save_traveler(user_uid, ds_traveler)
            log.debug('user={}: saved ds_traveler \r\n{}'.format(user_uid, ds_traveler))

        passenger_grouping = {}
        for order in orders:
            for passenger in order.passengers:
                group = PassengerGroup(order, passenger)
                key = group.get_key()
                passenger_grouping.setdefault(key, group)
                passenger_grouping[key].orders.append(order)
        log.debug('user={}: {} passengers found'.format(user_uid, len(passenger_grouping)))

        order_info_cache = {}
        for group in passenger_grouping.values():
            order = group.default_order
            if order.uid in order_info_cache:
                order_info = order_info_cache[order.uid]
            else:
                order_info = order_info_cache.setdefault(order.uid, get_order_info(order))
            group.set_order_info(order_info)
        log.debug('user={}: {} orders received'.format(user_uid, len(order_info_cache)))

        for group in passenger_grouping.values():
            ds_passenger = group.create_datasync_passenger(ds_traveler.id)
            ds_passenger = data_sync_client.save_passenger(user_uid, ds_passenger)
            log.debug('user={}: saved ds_passenger \r\n{}'.format(user_uid, ds_passenger))
            ds_doc = group.create_datasync_document(ds_passenger.id)
            ds_doc = data_sync_client.save_document(user_uid, ds_passenger, ds_doc)
            log.debug('user={}: saved ds_doc \r\n{}'.format(user_uid, ds_doc))

        log.debug('user={}: done'.format(user_uid))
        return True
    except Exception:
        log.exception('user={}: Ошибка при миграции в datasync'.format(user_uid))
        return False


@try_hard(max_retries=3, sleep_duration=10)  # при неудаче спит дольше, на случай кратковременных проблем в ИМ
@request_limiter.throttle
def get_order_info(order):
    """
    :rtype OrderInfoResult
    """
    return get_order_info_original(order)


def run_sync():
    user_uids = get_user_uids_to_sync()
    log.info('Start. {} users found'.format(len(user_uids)))
    i = 0
    for uid in user_uids:
        i += 1
        log.info('Processing user {} of {}'.format(i, len(user_uids)))
        get_orders_by_user(uid).update(set__sync_personal_data_status=SyncStatus.STARTED)
        if sync_user_orders(uid, list(get_orders_by_user(uid))):
            get_orders_by_user(uid).update(set__sync_personal_data_status=SyncStatus.DONE)
        else:
            get_orders_by_user(uid).update(set__sync_personal_data_status=SyncStatus.FAILED)
    log.info('Done.')
