# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import base64
import json
import logging
from datetime import datetime, timedelta

from django.test import Client

from common.apps.train_order.models import BackofficeUser
from travel.rasp.train_api.train_purchase.core.models import TrainOrder

log = logging.getLogger(__name__)


def run(date_from, date_to=datetime.now().date(), max_orders_count=30, page_size=50):
    log.info('start benchmark with params: date_from={}, date_to={}, max_orders_count={}, page_size={}'.format(
        date_from, date_to, max_orders_count, page_size
    ))
    backoffice_client = Client()
    user = BackofficeUser.objects.filter(is_active=True, is_admin=True)[0]
    auth_header = base64.b64encode(json.dumps({'type': 'direct', 'login': user.username}))
    backoffice_client.defaults['HTTP_AUTHORIZATION'] = auth_header
    backoffice_client.user = user
    uids = [order.uid for order in TrainOrder.objects.filter(
        reserved_to__gte=date_from, reserved_to__lte=date_to
    ).order_by('-reserved_to')[:max_orders_count]]
    mongo_elapsed = timedelta()
    ydb_elapsed = timedelta()
    total_reads = 0
    total_errors = 0
    total_fails = 0
    for uid in uids:
        cur_offset = 0
        cur_rows = page_size
        try:
            while cur_rows == page_size:
                check_result = check_one(backoffice_client, uid, page_size, cur_offset)
                if check_result:
                    cur_rows, cur_mongo_elapsed, cur_ydb_elapsed = check_result
                    mongo_elapsed += cur_mongo_elapsed
                    ydb_elapsed += cur_ydb_elapsed
                    total_reads += 1
                else:
                    cur_rows = 0
                    total_fails += 1
                cur_offset += page_size
        except Exception:
            log.exception('Error checking order={}'.format(uid))
            total_errors += 1
    log.info('Done: mongo_elapsed={}, ydb_elapsed={}, total_reads={}, total_errors={}, total_fails={}'.format(
        mongo_elapsed.total_seconds(), ydb_elapsed.total_seconds(), total_reads, total_errors, total_fails
    ))
    return mongo_elapsed, ydb_elapsed, total_reads, total_errors, total_fails


def check_one(backoffice_client, order_uid, limit, offset):
    params = {
        'orderUID': order_uid,
        'limit': limit,
        'offset': offset,
        'mode': 'mongo'
    }
    start = datetime.now()
    mongo_result = backoffice_client.get('/ru/train-purchase-backoffice/order-logs/', params).data
    mongo_elapsed = datetime.now() - start
    params['mode'] = 'ydb'
    start = datetime.now()
    ydb_result = backoffice_client.get('/ru/train-purchase-backoffice/order-logs/', params).data
    ydb_elapsed = datetime.now() - start
    if len(mongo_result['results']) != len(ydb_result['results']):
        log.warning('Mongo returns {0} rows, but ydb returns {1} rows. order_uid={2}, limit={3}, offset={4}'.format(
            len(mongo_result['results']), len(ydb_result['results']), order_uid, limit, offset
        ))
        return None
    return len(mongo_result['results']), mongo_elapsed, ydb_elapsed
