# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from marshmallow import fields, post_load, Schema

from common.models.geo import Station
from common.serialization.schema import get_defaults_from_schema
from common.utils.date import UTC_TZ
from common.utils.namedtuple import namedtuple_with_defaults
from common.utils.railway import get_railway_tz_by_point
from travel.rasp.train_api.serialization.schema_bases import PointsQuerySchema
from travel.rasp.train_api.wizard_api.helpers.dt import get_dt_from_wizard_dict

DATETIME_UTC_ISOFORMAT = '%Y-%m-%dT%H:%M:%SZ'


class TrainSegmentQuerySchema(PointsQuerySchema):
    number = fields.String(required=True)
    national_version = fields.String(default='ru')
    departure = fields.DateTime(required=True)

    @post_load
    def make_query(self, data):
        return TrainSegmentQuery(**data)


class TrainSegmentQuery(namedtuple_with_defaults(
    'OrderSegmentQuery', TrainSegmentQuerySchema().fields.keys(), get_defaults_from_schema(TrainSegmentQuerySchema))
):
    pass


class RawWizardBrandSchema(Schema):
    short_title = fields.String(dump_to='shortTitle')
    title = fields.String()
    is_deluxe = fields.Boolean(dump_to='isDeluxe')
    is_high_speed = fields.Boolean(dump_to='isHighSpeed')


class RawWizardTrainSchema(Schema):
    title = fields.String()
    number = fields.String()
    deluxe_train = fields.Nested(RawWizardBrandSchema, attribute='brand', dump_to='deluxeTrain')
    first_country_code = fields.String(dump_to='firstCountryCode')
    last_country_code = fields.String(dump_to='lastCountryCode')


class RawWizardTrainSegmentSchema(Schema):
    arrival = fields.Method(serialize='make_arrival')
    departure = fields.Method(serialize='make_departure')
    title = fields.Function(serialize=lambda s: s['train']['title'])
    number = fields.Function(serialize=lambda s: s['train']['display_number'] or s['train']['number'])
    duration_seconds = fields.Function(serialize=lambda s: s['duration'] * 60, dump_to='duration')
    has_dynamic_pricing = fields.Function(serialize=lambda s: s['train']['has_dynamic_pricing'],
                                          dump_to='hasDynamicPricing')
    thread = fields.Nested(RawWizardTrainSchema, attribute='train')
    company = fields.Method(serialize='make_company')
    station_from = fields.Method(serialize='make_station_from', dump_to='stationFrom')
    station_to = fields.Method(serialize='make_station_to', dump_to='stationTo')
    # для обратной совместимости с trains-front
    transport = fields.Constant(constant={'code': 'train', 'id': 1, 'title': 'Поезд'})

    def make_arrival(self, raw_segment):
        return get_dt_from_wizard_dict(raw_segment['arrival']['local_datetime'])\
            .astimezone(UTC_TZ).strftime(DATETIME_UTC_ISOFORMAT)

    def make_departure(self, raw_segment):
        return get_dt_from_wizard_dict(raw_segment['departure']['local_datetime'])\
            .astimezone(UTC_TZ).strftime(DATETIME_UTC_ISOFORMAT)

    def make_company(self, raw_segment):
        title = '/'.join(raw_segment['train']['coach_owners'] or [''])
        return {
            'shortTitle': title,
            'title': title,
        }

    def make_station_to(self, raw_segment):
        return self._make_station(raw_segment, event='arrival')

    def make_station_from(self, raw_segment):
        return self._make_station(raw_segment, event='departure')

    @staticmethod
    def _make_codes(station):
        express = station.get_code('express')
        if not express:
            return None
        return {'express': express}

    @staticmethod
    def _make_country(station):
        country = station.country
        if not country:
            return None
        return {
            'id': country.id,
            'code': country.code
        }

    def _make_station(self, raw_segment, event):
        station = raw_segment[event]['station']
        title = station['title']
        station_id = int(station['key'][1:])
        station = Station.objects.get(id=station_id)
        railway_tz = get_railway_tz_by_point(station)
        timezone = raw_segment[event]['local_datetime']['timezone']
        codes = RawWizardTrainSegmentSchema._make_codes(station)
        country = RawWizardTrainSegmentSchema._make_country(station)
        return {
            'shortTitle': title,
            'title': title,
            'id': station_id,
            'timezone': timezone,
            # для обратной совместимости с trains-front
            'country': country,
            'railwayTimezone': railway_tz.zone,
            'codes': codes
        }
