# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import re
from collections import namedtuple

from rest_framework import status
from rest_framework.decorators import api_view
from rest_framework.response import Response

from common.data_api.train_api.tariffs.utils import get_reverse_train_number
from travel.rasp.train_api.search.serialization import TrainSegmentQuerySchema, RawWizardTrainSegmentSchema
from travel.rasp.train_api.wizard_api.client import train_wizard_api_client

TRAIN_NUMBER_RE = re.compile(r'(\d+)(\D+)$')

TrainNumber = namedtuple('TrainNumber', ['digits', 'letters'])


@api_view(['GET'])
def train_segment(request):
    """
    Возвращает сегмент для отображения страницы заказа.
    ---
    parameters:
      - in: query
        name: query_params
        schema:
            $ref: 'TrainSegmentQuerySchema'
    responses:
        200:
            description: Сегмент
            schema:
                properties:
                    result:
                        $ref: 'RawWizardTrainSegmentSchema'
        400:
            schema:
                properties:
                    errors:
                        type: object
    """
    query, errors = TrainSegmentQuerySchema().load(request.GET)
    if errors:
        return Response({
            'errors': errors
        }, status=status.HTTP_400_BAD_REQUEST)

    raw_segment = get_raw_train_segment_from_wizard(query)
    if not raw_segment:
        return Response({
            'errors': {'wrongRequest': ['no_such_segment']}
        }, status=status.HTTP_400_BAD_REQUEST)

    return Response({'result': RawWizardTrainSegmentSchema().dump(raw_segment).data})


def get_raw_train_segment_from_wizard(query):
    departure = query.point_from.pytz.localize(query.departure)
    raw_result = train_wizard_api_client.get_raw_segments_with_tariffs(
        query.point_from.point_key,
        query.point_to.point_key,
        departure.date()
    )
    if not raw_result:
        return None

    raw_segments = raw_result['segments']
    query_number = parse_train_number(query.number)
    for raw_segment in raw_segments:
        raw_segment['train']['parsed_number'] = parse_train_number(raw_segment['train']['number'])
    raw_segments_with_same_number = [s for s in raw_segments
                                     if is_same_number(s['train']['parsed_number'], query_number)]
    if not raw_segments_with_same_number:
        return None

    return max(raw_segments_with_same_number, key=lambda s: (
        set(s['train']['parsed_number'].letters) == set(query_number.letters),  # буквы совпадают
        bool(s['train']['display_number']),  # если нет display_number, значит нет данных от партнера
        s['train']['thread_type'] != 'through_train',  # беспересадочные нитки в конце
        -s['duration'],  # приоретет самому быстрому сегменту
    ))


def is_same_number(first, second):
    has_same_digits = first.digits in (second.digits, get_reverse_train_number(second.digits))
    has_common_letter = set(first.letters) & set(second.letters)
    return has_same_digits and has_common_letter


def parse_train_number(train_number):
    match = TRAIN_NUMBER_RE.match(train_number)
    if not match:
        return None
    return TrainNumber(*match.groups())
