# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from contextlib import contextmanager
from enum import Enum

from common.models.geo import Settlement
from common.utils.title_generator import DASH
from common.dynamic_settings.default import conf

log = logging.getLogger(__name__)

USSR_CODES = {'AB', 'AZ', 'AM', 'BY', 'GE', 'KZ', 'KG', 'LV', 'LT', 'MD', 'RU', 'TJ', 'TM', 'UZ', 'UA', 'EE'}
INTERNATIONAL_CODES = {'AT', 'CZ', 'DE', 'FR', 'IT', 'MC', 'PL'}


class CountryType(Enum):
    USSR = 'u'
    INTERNATIONAL = 'i'
    FORBIDDEN = 'f'


class CountryAvailabilityManager(object):
    def __init__(self):
        self.title_cache = None

    def is_allowed_settlement_title(self, settlement_title, allow_international_routes):
        if self.title_cache is None:
            return None

        allowed_type = self.title_cache.get(settlement_title.lower())
        return (allowed_type == CountryType.USSR
                or (allow_international_routes and allowed_type == CountryType.INTERNATIONAL))

    def precache(self):
        def _get_country_type(country_code):
            if country_code in USSR_CODES:
                return CountryType.USSR
            elif country_code in INTERNATIONAL_CODES:
                return CountryType.INTERNATIONAL
            return None

        ussr_title_cache = {}
        international_title_cache = {}

        for settlement in Settlement.objects.filter(hidden=False):
            title = settlement.L_title(lang='ru')
            if not title:
                continue
            title = title.lower()

            if settlement.country:
                country_type = _get_country_type(settlement.country.code)
            else:
                log.warning("Settlement {} {} has not country".format(settlement.id, title))
                country_type = CountryType.FORBIDDEN

            if country_type == CountryType.USSR:
                ussr_title_cache.setdefault(title, country_type)
            elif country_type == CountryType.INTERNATIONAL:
                international_title_cache.setdefault(title, country_type)
            else:
                ussr_title_cache[title] = CountryType.FORBIDDEN

        self.title_cache = dict(filter(lambda e: e[1] == CountryType.USSR, ussr_title_cache.items()))
        self.title_cache.update(international_title_cache)

    @contextmanager
    def using_precache(self):
        if self.title_cache is None:
            self.precache()
            try:
                yield
            finally:
                self.title_cache = None
        else:
            yield

    @classmethod
    def is_allowed_country_code(cls, country_code, allow_international_routes):
        return country_code in USSR_CODES or (allow_international_routes and country_code in INTERNATIONAL_CODES)


def check_train_allowed(first_country_code, last_country_code, train_title, allow_international_routes=False):
    if (
        conf.TRAIN_PURCHASE_CHECK_IN_USSR_BY_TRAIN_TITLE
        and first_country_code is None
        and last_country_code is None
        and train_title
    ):
        title_route = [s.strip() for s in train_title.split(DASH)]
        if len(title_route) != 2:
            return False
        first_station, last_station = title_route
        return (country_availability_manager.is_allowed_settlement_title(first_station, allow_international_routes)
                and country_availability_manager.is_allowed_settlement_title(last_station, allow_international_routes))
    else:
        first_country_allowed = country_availability_manager.is_allowed_country_code(first_country_code,
                                                                                     allow_international_routes)
        last_country_allowed = country_availability_manager.is_allowed_country_code(last_country_code,
                                                                                    allow_international_routes)
        return first_country_allowed and last_country_allowed


country_availability_manager = CountryAvailabilityManager()
