# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import six
from django.conf import settings
from django.utils.encoding import force_text
from django.utils.functional import cached_property
from enum import Enum

from common.apps.train.tariff_error import TariffError
from common.apps.train_order.enums import COACH_TYPE_TO_IM_CAR_TYPE
from common.models.currency import Price
from common.models.transport import TransportType
from common.models_utils.i18n import RouteLTitle
from common.utils.namedtuple import namedtuple_with_defaults
from common.utils.railway import get_railway_tz_by_express_code, get_railway_tz_by_point
from travel.rasp.train_api.tariffs.train.base.availability_indication import AvailabilityIndication
from travel.rasp.train_api.tariffs.train.base.utils import get_point_express_code
from travel.rasp.train_api.train_purchase.utils.fee_calculator import calculate_ticket_cost

AVAILABILITY_INDICATION_TO_TARIFF_ERROR = {
    AvailabilityIndication.NOT_AVAILABLE_IN_WEB: TariffError.NOT_AVAILABLE_IN_WEB,
    AvailabilityIndication.FEATURE_NOT_ALLOWED: TariffError.FEATURE_NOT_ALLOWED,
    AvailabilityIndication.SERVICE_NOT_ALLOWED: TariffError.SERVICE_NOT_ALLOWED,
    AvailabilityIndication.CARRIER_NOT_ALLOWED_FOR_SALES: TariffError.CARRIER_NOT_ALLOWED_FOR_SALES,
    AvailabilityIndication.OTHER_REASON_OF_INACCESSIBILITY: TariffError.OTHER_REASON_OF_INACCESSIBILITY,
}


def check_tariff_coach_type(coach_type):
    return coach_type in COACH_TYPE_TO_IM_CAR_TYPE


def check_place_reservation_types(place_reservation_type):
    return place_reservation_type in {PlaceReservationType.USUAL,
                                      PlaceReservationType.TWO_PLACES_AT_ONCE,
                                      PlaceReservationType.FOUR_PLACES_AT_ONCE}


class PlaceReservationType(Enum):
    UNKNOWN = 'unknown'  # неизвестное
    USUAL = 'usual'  # выкуп мест по одному
    TWO_PLACES_AT_ONCE = 'two_places_at_once'  # выкуп двух мест сразу - "мягкие" вагоны, часть СВ в Стрижах
    FOUR_PLACES_AT_ONCE = 'four_places_at_once'  # выкуп четырех мест сразу - купе-переговорные в Сапсанах


@six.python_2_unicode_compatible
class TrainTariff(object):
    order_url = None
    fee = None
    fee_percent = None
    is_bandit_fee_applied = False
    bandit_type = None
    bandit_version = None

    def __init__(self, coach_type, ticket_price, service_price,
                 seats=1, lower_seats=0, upper_seats=0,
                 lower_side_seats=0, upper_side_seats=0,
                 max_seats_in_the_same_car=1, several_prices=False,
                 place_reservation_type=PlaceReservationType.USUAL,
                 is_transit_document_required=False,
                 availability_indication=AvailabilityIndication.UNKNOWN,
                 service_class=None,
                 category_traits=None,
                 has_non_refundable_tariff=False):
        self.real_coach_type = coach_type
        self.ticket_price = ticket_price
        self.service_price = service_price
        self.seats = seats
        self.lower_seats = lower_seats
        self.upper_seats = upper_seats
        self.lower_side_seats = lower_side_seats
        self.upper_side_seats = upper_side_seats
        self.max_seats_in_the_same_car = max_seats_in_the_same_car
        self.several_prices = several_prices
        self.place_reservation_type = place_reservation_type
        self.is_transit_document_required = is_transit_document_required
        self.availability_indication = availability_indication
        self.service_class = service_class
        self.category_traits = category_traits
        self.has_non_refundable_tariff = has_non_refundable_tariff

    def calculate_fee(self, contract, yandex_uid=None):
        if not self.ticket_price:
            self.fee = self.ticket_price
            return

        tariff = self.ticket_price.value
        service_tariff = self.service_price.value if self.service_price else 0
        ticket_cost = calculate_ticket_cost(contract, self.coach_type, tariff, service_tariff, yandex_uid=yandex_uid)
        self.fee = Price(ticket_cost.full_fee, self.ticket_price.currency)
        self.fee_percent = ticket_cost.yandex_fee_percent

    @property
    def coach_type(self):
        return self.real_coach_type.value

    @property
    def total_price(self):
        if self.fee is None:
            raise AttributeError('call calculate_fee before using total_price')
        return self.ticket_price + self.fee

    def validate(self):
        errors = set()
        if not check_tariff_coach_type(self.real_coach_type):
            errors.add(TariffError.UNSUPPORTED_COACH_TYPE.value)
        if self.ticket_price.value <= 10:
            errors.add(TariffError.TOO_CHEAP.value)
        if not check_place_reservation_types(self.place_reservation_type):
            errors.add(TariffError.UNSUPPORTED_RESERVATION_TYPE.value)
        if self.seats <= 0:
            errors.add(TariffError.SOLD_OUT.value)
        if self.is_transit_document_required:
            errors.add(TariffError.TRANSIT_DOCUMENT_REQUIRED.value)
        if self.category_traits is not None and any(t.is_for_children for t in self.category_traits):
            errors.add(TariffError.CHILD_TARIFF.value)
        if self.availability_indication != AvailabilityIndication.AVAILABLE:
            errors.add(AVAILABILITY_INDICATION_TO_TARIFF_ERROR.get(
                self.availability_indication, TariffError.UNKNOWN).value)

        return len(errors) == 0, errors

    def __eq__(self, other):
        if isinstance(other, self.__class__):
            return self.__dict__ == other.__dict__
        return False

    def __str__(self):
        return '<TrainTariff: \n  {}\n>'.format('\n  '.join(
            ('{}: {}'.format(k, force_text(v))) for k, v in self.__dict__.items()
        ))

    def __repr__(self):
        return self.__str__()


@six.python_2_unicode_compatible
class TrainSegment(object):
    key = None
    original_number = None  # Номер поезда из N1
    number = None  # Номер для отображения пользователю N2
    title = None
    title_common = None
    thread = None
    thread_variant = None
    search_segment = None
    t_model = None
    station_from = None
    station_from_ufs_title = None
    station_from_express_code = None
    station_to = None
    station_to_ufs_title = None
    station_to_express_code = None
    possible_numbers = None
    can_supply_segments = None  # Можно ли использовать сегмент для пополнения
    start_express_title_or_code = None
    end_express_title_or_code = None
    railway_departure = None
    railway_arrival = None
    departure = None
    arrival = None
    is_deluxe = None
    coach_owners = ()
    raw_train_category = None
    raw_train_name = None
    has_dynamic_pricing = False
    two_storey = False
    is_suburban = False
    start_station = None
    first_country_code = None
    end_station = None
    last_country_code = None
    ufs_title = None
    provider = None

    updated_at = None  # Время получения информации, актуально только для сегментов из колдунщика

    def __init__(self):
        self.t_type = TransportType.objects.get(pk=TransportType.TRAIN_ID)
        self.tariffs = {}
        self.L_title = RouteLTitle(self)

    def __str__(self):
        return '<TrainSegment: \n  {}\n>'.format('\n  '.join(
            ('{}: {}'.format(k, force_text(v))) for k, v in self.__dict__.items()
        ))

    @property
    def company(self):
        return self.thread and self.thread.company

    @cached_property
    def duration(self):
        return self.arrival - self.departure

    def get_duration(self):
        return int(self.duration.total_seconds() // 60)


@six.python_2_unicode_compatible
class TrainTariffsQuery(namedtuple_with_defaults(
    'TrainTariffsQuery',
    (
        'partner', 'departure_point',
        'arrival_point', 'departure_date',
        'include_reason_for_missing_prices', 'icookie',
        'bandit_type', 'original_query', 'mock_im_path', 'mock_im_auto',
        'train_api_use_worker'
    ),
)):
    @cached_property
    def departure_point_code(self):
        return get_point_express_code(self.departure_point)

    @cached_property
    def arrival_point_code(self):
        return get_point_express_code(self.arrival_point)

    @cached_property
    def departure_railway_tz(self):
        return (get_railway_tz_by_express_code(self.departure_point_code) or
                get_railway_tz_by_point(self.departure_point))

    @cached_property
    def arrival_railway_tz(self):
        return (get_railway_tz_by_express_code(self.arrival_point_code) or
                get_railway_tz_by_point(self.arrival_point))

    @property
    def mock_im(self):
        return (self.mock_im_path or self.mock_im_auto) and settings.ENABLE_MOCK_IM

    @property
    def cache_key(self):
        result = settings.CACHEROOT + ('{self.partner.value}_tariffs/'
                                       '{self.departure_point_code}_{self.arrival_point_code}/'
                                       '{self.departure_date:%Y%m%d}/'
                                       'r{self.include_reason_for_missing_prices}').format(self=self)
        if self.mock_im:
            result = result + '/' + (self.mock_im_path if self.mock_im_path else 'mock-im-auto')
        return result

    def __str__(self):
        return force_text(super(TrainTariffsQuery, self).__repr__())


class TrainMinPrices(object):
    is_suburban = None
    has_dynamic_pricing = None
    two_storey = None
    original_number = None
    display_number = None
    tariffs = {}
    departure = None
    arrival = None
    departure_station_id = None
    arrival_station_id = None
    coach_owners = []
