# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from collections import namedtuple
from django.conf import settings

from common.dynamic_settings.default import conf
from travel.rasp.library.python.common23.date import environment
from common.utils.date import daterange
from common.utils.iterrecipes import unique_everseen
from common.utils.railway import get_railway_tz_by_point
from travel.rasp.train_api.tariffs.train.base.models import TrainTariffsQuery
from travel.rasp.train_api.tariffs.train.base.query_result_manager import QueryResultManager
from travel.rasp.train_api.tariffs.train.base.utils import build_order_urls
from travel.rasp.train_api.tariffs.train.base.worker import TrainTariffsResult
from travel.rasp.train_api.tariffs.train.im.send_query import do_im_query, postprocess_query_result
from travel.rasp.train_api.tariffs.train.wizard.service import get_wizard_tariffs
from travel.rasp.train_api.train_purchase.core.models import TrainPartner


def get_train_segments(
        point_from, point_to, dt_from, dt_to, national_version, can_ask_partner,
        include_price_fee, yandex_uid, force_ufs_order, allow_international_routes,
        icookie, bandit_type, asker, ytp_referer, train_api_use_worker, original_query,
):
    """
    :param point_from: Город или Станция
    :param point_to: Город или Станция
    :param dt_from: Начало интервала, aware datetime.
    :param dt_to: Конец интервала не включительно.
    :param national_version: Национальная версия
    :param include_price_fee: Включать наш сбор
    :param can_ask_partner: Можно ли делать запросы к партнеру
    :param yandex_uid: Uid пользователя для спецобработки
    :param force_ufs_order: Принудительный заказ с УФС
    :param allow_international_routes: Разрешить выдачу международных направлений
    :param icookie: Расширенный Uid пользователя для спецобработки
    :param bandit_type: Тип "бандита" для экспериментов с наценкой
    :param asker: признак запроса от пересадочника
    :param ytp_referer: признак внутреннего перехода для happy page
    :param train_api_use_worker: ключ необходимости похода в worker для эксперимента
    :param original_query: исходный запрос TrainQuerySchema
    :return TrainSegmentsResult
    """
    include_reason_for_missing_prices = conf.TRAIN_PURCHASE_FEATURE_REASON_FOR_MISSING_PRICES
    get_results = (
        get_results_with_missing_prices
        if include_reason_for_missing_prices
        else get_price_results
    )

    results = get_results(
        point_from, point_to, dt_from, dt_to, can_ask_partner,
        yandex_uid=yandex_uid,
        include_reason_for_missing_prices=include_reason_for_missing_prices,
        icookie=icookie,
        bandit_type=bandit_type,
        asker=asker,
        train_api_use_worker=train_api_use_worker,
        original_query=original_query,
    )

    querying = any(r.status == TrainTariffsResult.STATUS_PENDING for r in results)

    segments = [_s for _r in results for _s in _r.segments]
    segments = filter_empty_segments(segments, include_reason_for_missing_prices)
    segments.sort(key=lambda s: s.departure)
    # могут быть одинаковые сегменты, отфильтровываем дубли
    segments = list(unique_everseen(segments, key=lambda s: (s.departure, s.arrival, s.original_number, s.ufs_title, s.key)))

    def in_time_range(sg):
        return dt_from <= sg.departure <= dt_to

    segments = list(filter(in_time_range, segments))

    for segment in segments:
        build_order_urls(
            segment, point_from, point_to, national_version,
            force_ufs_order=force_ufs_order,
            allow_international_routes=allow_international_routes,
            ytp_referer=ytp_referer,
        )

        for tariff_info in segment.tariffs['classes'].values():
            tariff_info.price = tariff_info.total_price if include_price_fee else tariff_info.ticket_price
    return TrainSegmentsResult(segments, querying)


def _get_wizard_price_results(tariff_query, yandex_uid):
    wizard_result = get_wizard_tariffs(tariff_query, expired_timeout=settings.PATHFINDER_EXPIRED_TIMEOUT_MINUTES)
    if wizard_result:
        postprocess_query_result(wizard_result, tariff_query, yandex_uid=yandex_uid)
    return wizard_result


def get_price_results(point_from, point_to, dt_from, dt_to, can_ask_partner, yandex_uid,
                      include_reason_for_missing_prices, icookie, bandit_type, asker, train_api_use_worker, original_query):
    results = []

    queries = [
        q for q in make_train_queries(
            point_from, point_to, dt_from, dt_to,
            include_reason_for_missing_prices, icookie, bandit_type, original_query, train_api_use_worker,
        )
        if can_send_query(q)
    ]

    for tariff_query in queries:
        result = TrainTariffsResult.get_from_cache(tariff_query)

        if result:
            if asker == 'pathfinder':
                result_bound_time = result.expired_with_timeout(settings.PATHFINDER_EXPIRED_TIMEOUT_MINUTES)
            else:
                result_bound_time = result.expired_at

        if result and environment.now_aware() < result_bound_time:
            postprocess_query_result(result, tariff_query, yandex_uid=yandex_uid)
            results.append(result)
        else:
            result = None
            if asker == 'pathfinder':
                result = _get_wizard_price_results(tariff_query, yandex_uid) or result
                if (result is None or result.status == TrainTariffsResult.STATUS_PENDING) and can_ask_partner:
                    result = do_im_query(tariff_query, include_reason_for_missing_prices)
            else:
                if can_ask_partner:
                    result = do_im_query(tariff_query, include_reason_for_missing_prices)
                if result is None or result.status == TrainTariffsResult.STATUS_PENDING:
                    result = _get_wizard_price_results(tariff_query, yandex_uid) or result

            if result:
                results.append(result)

    return results


def get_results_with_missing_prices(
    point_from, point_to, dt_from, dt_to, can_ask_partner, yandex_uid,
    include_reason_for_missing_prices, icookie, bandit_type, asker, train_api_use_worker, original_query
):
    results = []

    queries = [
        q for q in make_train_local_queries(
            point_from, point_to, dt_from, dt_to,
            include_reason_for_missing_prices, icookie, bandit_type, original_query, train_api_use_worker
        ) if can_send_local_query(q)
    ]

    for tariff_query in queries:
        result = QueryResultManager(
            tariff_query, can_ask_partner, include_reason_for_missing_prices,
            asker == 'pathfinder'
        ).get_result()
        if result:
            postprocess_query_result(result, tariff_query, yandex_uid=yandex_uid)
            results.append(result)

    return results


def can_send_query(train_query):
    if not (train_query.departure_point_code and train_query.arrival_point_code and train_query.departure_railway_tz):
        return False

    railway_today = environment.now_aware().astimezone(train_query.departure_railway_tz).date()
    return 0 <= (train_query.departure_date - railway_today).days <= conf.TRAIN_ORDER_DEFAULT_DEPTH_OF_SALES


def can_send_local_query(train_query):
    if not (train_query.departure_point_code and train_query.arrival_point_code and train_query.departure_railway_tz):
        return False

    local_today = environment.now_aware().astimezone(train_query.departure_point.pytz).date()
    return 0 <= (train_query.departure_date - local_today).days <= conf.TRAIN_ORDER_DEFAULT_DEPTH_OF_SALES


def _iter_dates(min_dt, max_dt):
    return daterange(min_dt.date(), max_dt.date(), include_end=bool(max_dt.time())) if min_dt < max_dt else []


def make_train_queries(
    departure_point, arrival_point, departure_min_dt, departure_max_dt,
    include_reason_for_missing_prices, icookie, bandit_type, original_query, train_api_use_worker=None
):
    railway_tz = get_railway_tz_by_point(departure_point)
    departure_min_railway_dt = departure_min_dt.astimezone(railway_tz)
    departure_max_railway_dt = departure_max_dt.astimezone(railway_tz)
    return [
        TrainTariffsQuery(
            TrainPartner.IM, departure_point, arrival_point,
            departure_date, include_reason_for_missing_prices, icookie, bandit_type, original_query,
            original_query.get('mock_im_path'), original_query.get('mock_im_auto'),
            train_api_use_worker
        ) for departure_date in _iter_dates(departure_min_railway_dt, departure_max_railway_dt)
    ]


def make_train_local_queries(
    departure_point, arrival_point, departure_min_dt, departure_max_dt,
    include_reason_for_missing_prices, icookie, bandit_type, original_query, train_api_use_worker=None
):
    local_tz = departure_point.pytz
    departure_min_local_dt = departure_min_dt.astimezone(local_tz)
    departure_max_local_dt = departure_max_dt.astimezone(local_tz)
    return [
        TrainTariffsQuery(
            TrainPartner.IM, departure_point, arrival_point, departure_date,
            include_reason_for_missing_prices, icookie, bandit_type,
            original_query, original_query.get('mock_im_path'), original_query.get('mock_im_auto'),
            train_api_use_worker
        ) for departure_date in _iter_dates(departure_min_local_dt, departure_max_local_dt)
    ]


def filter_empty_segments(segments, include_reason_for_missing_prices):
    # type: (List[TrainSegment]) -> List[TrainSegment]
    if not segments:
        return segments

    if include_reason_for_missing_prices:
        return [s for s in segments if s.tariffs.get('classes') or s.tariffs.get('broken_classes')]
    else:
        return [s for s in segments if s.tariffs.get('classes')]


TrainSegmentsResult = namedtuple('TrainSegmentsResult', ('segments', 'querying'))
