# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from django.conf import settings

from common.dynamic_settings.default import conf
from travel.rasp.library.python.common23.date import environment
from travel.rasp.train_api.helpers.rps_limiter import rps_limiter, RPS_LIMITER_IM_SEARCH_KEY
from travel.rasp.train_api.tariffs.train.base.worker import TrainTariffsResult
from travel.rasp.train_api.tariffs.train.im.send_query import do_im_query
from travel.rasp.train_api.tariffs.train.wizard.service import get_wizard_tariffs

log = logging.getLogger(__name__)


class QueryResultManager(object):
    def __init__(self, query, can_ask_partner, include_reason_for_missing_prices, is_pathfider_query=False):
        self.query = query
        self.can_ask_partner = can_ask_partner
        self.include_reason_for_missing_prices = include_reason_for_missing_prices

        self.status_storage = StatusStorage()
        self.status_result = None

        self.data_storage = DataStorage()
        self.need_data_from_data_storage = False
        self.data_result = None

        self.need_to_ask_partner = False
        self.is_pathfider_query = is_pathfider_query

    def get_result(self):
        self._get_status()
        self._make_decision()
        self._do_work()
        return self._get_result()

    def _get_status(self):
        self.status_result = self.status_storage.get_result(self.query)

    def _make_decision(self):
        self.need_data_from_data_storage = self._need_data_from_data_storage()
        self.need_to_ask_partner = self._need_to_ask_partner()

    def _need_data_from_data_storage(self):
        """
        Не надо доставать результат из хранилища (колдунщика) только если в статусе ошибка.
        """
        if (
            self.status_result is not None
            and self.status_result.status == TrainTariffsResult.STATUS_ERROR
        ):
            return False
        # mock-im ответы не храним в колдуне, только в кеше
        if self.query.mock_im:
            return False
        return True

    def _need_to_ask_partner(self):
        if not self.can_ask_partner:
            return False

        if self.status_result is None:
            return True

        expired_timeout = (
            self.status_result.expired_with_timeout(settings.PATHFINDER_EXPIRED_TIMEOUT_MINUTES)
            if conf.TRAIN_TARIFFS_ENABLE_EXPIRED_TIMEOUT_FOR_PATHFINDER and self.is_pathfider_query
            else self.status_result.expired_at
        )

        if expired_timeout < environment.now_aware():
            return True

        return False

    def _do_work(self):
        if self.need_data_from_data_storage:
            self.data_result = self.data_storage.get_result(self.query, self.is_pathfider_query)
            if not self.data_result:
                log.warning('Не смогли получить данные из хранилища (колдунщика) %s', self.query)
        if self.need_to_ask_partner and (
            not self.is_pathfider_query or
            not self.data_result or
            self.data_result.status == TrainTariffsResult.STATUS_PENDING
        ):
            query_allowed = True
            if conf.TRAIN_PURCHASE_IM_RPS_LIMITER_ENABLED and not self.query.mock_im:
                query_allowed = rps_limiter.check_query_allowed(RPS_LIMITER_IM_SEARCH_KEY,
                                                                conf.TRAIN_PURCHASE_IM_SEARCH_RPS_LIMIT,
                                                                conf.TRAIN_PURCHASE_IM_RPS_LIMITER_PERIOD)
            if query_allowed:
                self.status_result = self._ask_partner()

    def _ask_partner(self):
        # TODO: разнести логику получения данных и сохранения их в хранилища.
        #  Это можно сделать только после перехода на использовать memcached только для статусов
        return do_im_query(self.query, self.include_reason_for_missing_prices)

    def _get_result(self):
        if self.need_to_use_status_result(self.data_result, self.status_result):
            result = self.status_result
            result_label = 'force_status_result'
            extra_info = ''
            if self.data_result:
                extra_info = '. Unused data_result: {}'.format(self.data_result)
        elif self.data_result:
            self.add_supplement_segments(self.data_result, self.status_result)
            result = self.data_result
            result_label = 'data_result'
            extra_info = '. Has status_result: {}'.format(self.status_result)
        else:
            result = self.status_result
            result_label = 'status_result'
            extra_info = ''

        log.info('QueryResultManager done. Use %s: %s. Query: %s%s', result_label, result, self.query, extra_info)
        return result

    @staticmethod
    def add_supplement_segments(data_result, status_result):
        # TODO: Удалить, когда колдунщик сможет отдавать сегменты пополнения
        if not status_result or not status_result.segments:
            return

        if not data_result.segments:
            data_result.segments = []
        data_result.segments += [s for s in status_result.segments if not s.thread]

    @staticmethod
    def need_to_use_status_result(data_result, status_result):
        # TODO: Удалить, когда колдунщик будет правильно отдавать updated_at
        return (
            data_result
            and data_result.status == TrainTariffsResult.STATUS_PENDING
            and status_result
            and status_result.status == TrainTariffsResult.STATUS_SUCCESS
        )


class StatusStorage(object):
    def get_result(self, query):
        return TrainTariffsResult.get_from_cache(query)


class DataStorage(object):
    def get_result(self, query, is_pathfider_query):
        if conf.TRAIN_TARIFFS_ENABLE_EXPIRED_TIMEOUT_FOR_PATHFINDER and is_pathfider_query:
            return get_wizard_tariffs(query, expired_timeout=settings.PATHFINDER_EXPIRED_TIMEOUT_MINUTES)
        return get_wizard_tariffs(query)
