# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from functools import reduce
from itertools import groupby

from travel.rasp.train_api.tariffs.train.base.availability_indication import AvailabilityIndication
from travel.rasp.train_api.tariffs.train.base.models import TrainTariff, PlaceReservationType


def join_train_tariffs(tariff1, tariff2):
    """
    Объединяет два тарифа. Возвращает тариф минимальной ценой проезда.
    :type tariff1: train_api.tariffs.train.base.models.TrainTariff
    :type tariff2: train_api.tariffs.train.base.models.TrainTariff
    :rtype: train_api.tariffs.train.base.models.TrainTariff
    """
    if tariff1.coach_type != tariff2.coach_type:
        raise ValueError('Cannot join tariffs of different types.')

    if tariff1.ticket_price == tariff2.ticket_price:
        several_prices = tariff1.several_prices or tariff2.several_prices
        ticket_price = tariff1.ticket_price
        # Не начисляем комиссию на белье в плацкарте.
        # Поэтому с учетом комиссии при равных ticket_price более дешевым будет билет с бОльшим service_price
        max_service_tariff = max(tariff1, tariff2, key=lambda tariff: tariff.service_price)
        service_price = max_service_tariff.service_price
        service_class = max_service_tariff.service_class
    else:
        min_tariff = min(tariff1, tariff2, key=lambda tariff: tariff.ticket_price)
        several_prices = True
        ticket_price = min_tariff.ticket_price
        service_price = min_tariff.service_price
        service_class = min_tariff.service_class

    return TrainTariff(
        coach_type=tariff1.real_coach_type,
        ticket_price=ticket_price,
        service_price=service_price,
        service_class=service_class,
        seats=tariff1.seats + tariff2.seats,
        lower_seats=tariff1.lower_seats + tariff2.lower_seats,
        upper_seats=tariff1.upper_seats + tariff2.upper_seats,
        lower_side_seats=tariff1.lower_side_seats + tariff2.lower_side_seats,
        upper_side_seats=tariff1.upper_side_seats + tariff2.upper_side_seats,
        max_seats_in_the_same_car=max(
            tariff1.max_seats_in_the_same_car,
            tariff2.max_seats_in_the_same_car
        ),
        several_prices=several_prices,
        place_reservation_type=PlaceReservationType.USUAL,
        is_transit_document_required=False,
        availability_indication=AvailabilityIndication.AVAILABLE,
        has_non_refundable_tariff=tariff1.has_non_refundable_tariff or tariff2.has_non_refundable_tariff,
    )


def build_train_tariffs_classes(all_train_tariffs):
    tariff_classes = {}
    broken_tariff_classes = {}
    all_train_tariffs.sort(key=lambda x: x.coach_type)
    for coach_type, group in groupby(all_train_tariffs, key=lambda tariff: tariff.coach_type):
        group = list(group)
        tariff_classes_errors = set()
        valid_tariffs = []
        for tariff in group:
            valid, errors = tariff.validate()
            tariff_classes_errors |= errors
            if valid:
                valid_tariffs.append(tariff)

        if not valid_tariffs:
            broken_tariff_classes[coach_type] = list(tariff_classes_errors)
        else:
            tariff_classes[coach_type] = reduce(join_train_tariffs, valid_tariffs)

    return tariff_classes, broken_tariff_classes
