# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import re
from datetime import timedelta

from django.conf import settings
from django.utils.http import urlencode
from django.utils.lru_cache import lru_cache

from common.apps.train.tariff_error import TariffError
from common.apps.train_order.enums import CoachType
from common.models.geo import Station, Settlement, StationCode, StationMajority
from common.models.transport import TransportType
from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from common.utils.date import MSK_TZ
from common.utils.locations import set_lang_param
from travel.rasp.train_api.tariffs.train.base.country_availability_manager import check_train_allowed
from travel.rasp.train_api.train_purchase.core.enums import TrainOrderUrlOwner

define_setting('ALIGN_SEARCH_SEGMENT_KEYS', {Configuration.PRODUCTION: True}, default=False)

TIME_RANGE_HOURS = 2
NUMBER_FORMAT = '{:03}{}'
TRAIN_KEY_FORMAT = 'train {} {}'

train_number_re = re.compile(r'(\d+)(\D*)$')

CPPK_CARRIER = 'ЦППК'


def build_order_urls(train_segment, point_from, point_to, national_version, force_ufs_order=False,
                     allow_international_routes=False, ytp_referer=None):
    for tariff_class, tariff in train_segment.tariffs['classes'].items():
        tariff.order_url = build_old_morda_order_url(train_segment, tariff_class, tariff, point_from, point_to,
                                                     national_version)
        tariff.train_order_url, tariff.train_order_url_owner = _build_train_order_url(
            train_segment, tariff, point_from, point_to, force_ufs_order, allow_international_routes, ytp_referer
        )


def _build_train_order_url(train_segment, tariff, point_from, point_to, force_ufs_order=False,
                           allow_international_routes=False, ytp_referer=None):
    return build_train_order_url(
        train_segment.departure,
        train_segment.original_number,
        train_segment.number,
        getattr(train_segment, 'old_ufs_order', False),
        # TODO можем брать first_country_code и last_country_code из train_segment, когда сможем на странице покупки
        #  получать нужный сегмент https://st.yandex-team.ru/TRAINS-5277#5e4bbddfe281925daf6ba0c6
        train_segment.thread.first_country_code if train_segment.thread else None,
        train_segment.thread.last_country_code if train_segment.thread else None,
        tariff.coach_type,
        point_from,
        point_to,
        train_segment.title,
        force_ufs_order=force_ufs_order,
        allow_international_routes=allow_international_routes,
        provider=train_segment.provider,
        ytp_referer=ytp_referer,
    )


def build_train_order_url(departure, number, segment_number, old_ufs_order, first_country_code, last_country_code,
                          coach_type, point_from, point_to, train_title, force_ufs_order=False,
                          allow_international_routes=False, provider=None, ytp_referer=None):
    if not force_ufs_order and can_sale_on_rasp(old_ufs_order, first_country_code, last_country_code,
                                                point_from, point_to, train_title, allow_international_routes):
        departure_local = departure.astimezone(point_from.pytz).replace(tzinfo=None)
        params = {
            'fromId': point_from.point_key,
            'fromName': point_from.L_title(),
            'toId': point_to.point_key,
            'toName': point_to.L_title(),
            'when': departure_local.strftime('%Y-%m-%d'),
            'time': departure_local.strftime('%H:%M'),
            'number': number,
            'coachType': coach_type,
        }
        if provider:
            params['provider'] = provider
        if ytp_referer is not None:
            params['ytp_referer'] = ytp_referer

        return '/order/?{}'.format(urlencode(params, doseq=True)), TrainOrderUrlOwner.TRAINS
    else:
        from travel.rasp.train_api.train_partners.ufs.reserve_tickets import ADVERT_DOMAIN
        params = {
            'date': departure.astimezone(MSK_TZ).replace(tzinfo=None).strftime('%d.%m.%Y'),
            'domain': ADVERT_DOMAIN,
            'trainNumber': segment_number,
        }
        return 'https://{host}/kupit-zhd-bilety/{point_from}/{point_to}?{params}'.format(
            host=settings.UFS_HOST, params=urlencode(params, doseq=True),
            point_from=get_point_express_code(point_from), point_to=get_point_express_code(point_to),
        ), TrainOrderUrlOwner.UFS


def can_sale_on_rasp(old_ufs_order, first_country_code, last_country_code, point_from, point_to, train_title,
                     allow_international_routes=False):
    if old_ufs_order:
        return False
    if not check_train_allowed(first_country_code, last_country_code, train_title, allow_international_routes):
        return False
    if (point_from.country and point_from.country.code == 'UA') or (point_to.country and point_to.country.code == 'UA'):
        return False
    return True


def build_old_morda_order_url(train_segment, tariff_class, tariff, point_from, point_to, national_version):
    point_from = (train_segment.station_from if train_segment.station_from else point_from)
    point_to = (train_segment.station_to if train_segment.station_to else point_to)

    params = {
        'station_from': train_segment.station_from and train_segment.station_from.id,
        'station_to': train_segment.station_to and train_segment.station_to.id,
        'point_from': point_from.point_key,
        'point_to': point_to.point_key,
        'departure': train_segment.departure.astimezone(MSK_TZ).replace(tzinfo=None),
        'arrival': train_segment.arrival.astimezone(MSK_TZ).replace(tzinfo=None),
        'title': train_segment.L_title(),
        'date': train_segment.departure.date(),
        'number': train_segment.original_number,
        't_type': train_segment.t_type.code,
        'cls': tariff_class,
        'tariff': tariff.ticket_price.value,
    }

    if train_segment.tariffs['electronic_ticket']:
        params['et'] = 't'

    if train_segment.thread and hasattr(train_segment.thread, 'uid'):
        params['thread'] = train_segment.thread.uid

    return '/buy/?' + set_lang_param(params, national_version=national_version)


def make_segment_train_keys(segment):
    """
    Генерируем несколько ключей, чтобы привязываться к рейсам в пределах 4 часов.
    В том случае, если вдруг наше время отличается от времени UFS
    """

    if not (segment.thread and segment.thread.t_type_id == TransportType.TRAIN_ID):
        return []

    numbers = get_possible_numbers(segment.number)
    if getattr(segment, 'letters', None) and getattr(segment, 'train_number', None):
        digits = [segment.train_number.digits, get_reverse_train_number(segment.train_number.digits)]
        numbers = ['{}{}'.format(digit, letter) for letter in segment.letters for digit in digits]

    return _make_datetime_keys(segment, numbers)


def make_segment_suburban_express_keys(segment):
    """
    Генерируем ключи из поездов-дублей для электричек-экспрессов
    """

    if not (segment.thread and segment.thread.t_type_id == TransportType.SUBURBAN_ID):
        return []

    train_purchase_numbers = getattr(segment, 'train_purchase_numbers', [])
    if not train_purchase_numbers:
        return[]

    numbers = []
    for tpn in train_purchase_numbers:
        numbers += get_possible_numbers(tpn)

    return _make_datetime_keys(segment, numbers)


def _make_datetime_keys(segment, numbers):
    if settings.ALIGN_SEARCH_SEGMENT_KEYS:
        dt_keys = {
            make_datetime_key(segment.departure - timedelta(hours=TIME_RANGE_HOURS / 2)),
            make_datetime_key(segment.departure + timedelta(hours=TIME_RANGE_HOURS / 2))
        }
    else:
        dt_keys = {
            make_precise_datetime_key(segment.departure)
        }

    return [
        TRAIN_KEY_FORMAT.format(number, dt_key)
        for number in numbers
        for dt_key in sorted(dt_keys)
    ]


def make_tariff_segment_key(segment):
    return make_tariff_segment_key_exact(segment.original_number, segment.departure)


def make_tariff_segment_key_exact(number, departure):
    make_departure_key_func = make_datetime_key if settings.ALIGN_SEARCH_SEGMENT_KEYS else make_precise_datetime_key
    return TRAIN_KEY_FORMAT.format(number, make_departure_key_func(departure))


def find_city_train_station(city):
    codes_qs = StationCode.objects.filter(station__settlement=city,
                                          station__t_type__id=TransportType.TRAIN_ID,
                                          system__code='express').values_list('station__id', 'code')

    try:
        return codes_qs.filter(station__majority__id=StationMajority.EXPRESS_FAKE_ID)[0]
    except IndexError:
        pass

    try:
        return codes_qs.filter(station__hidden=False).order_by('station__majority__id')[0]
    except IndexError:
        pass

    return None, None


@lru_cache()
def get_point_express_code(point):
    if isinstance(point, Station):
        return point.get_code('express')

    if isinstance(point, Settlement):
        _station_id, express_code = find_city_train_station(point)
        return express_code

    return None


def get_reverse_train_number(train_digits):
    train_digits = int(train_digits)
    if train_digits % 2 == 0:
        train_digits -= 1
    else:
        train_digits += 1
    return train_digits


def get_possible_numbers(train_number):
    """
    Поезд может менять номер с четного на нечетный в зависимости от участка пути.
    Это необходимо учитывать при привязке цен, поэтому генерируем все возможные номера.
    """
    try:
        number, letter = _get_number_and_first_letter(train_number)
    except _StrangeTrainNumber:
        return [train_number]
    else:
        return [NUMBER_FORMAT.format(number, letter), NUMBER_FORMAT.format(get_reverse_train_number(number), letter)]


def make_datetime_key(dt):
    aligned_dt = dt.replace(hour=int(dt.hour // TIME_RANGE_HOURS) * TIME_RANGE_HOURS)
    return aligned_dt.strftime('%Y%m%d_%H')


def make_precise_datetime_key(dt):
    return dt.strftime('%Y%m%d_%H%M')


def make_time_key(dt):
    aligned_dt = dt.replace(hour=int(dt.hour // TIME_RANGE_HOURS) * TIME_RANGE_HOURS)
    return aligned_dt.strftime('%H')


def fix_train_number(train_number):
    """
    "*" встречается, например, в поиске Тюмень Сургут 2016-05-10, правда в теге N2.
    """
    train_number = train_number.replace('*', '')
    try:
        number, letter = _get_number_and_first_letter(train_number)
    except _StrangeTrainNumber:
        return train_number
    else:
        return NUMBER_FORMAT.format(number, letter)


class _StrangeTrainNumber(Exception):
    pass


def _get_number_and_first_letter(train_number):
    match = train_number_re.match(train_number)
    if not match:
        raise _StrangeTrainNumber

    number, letters = match.groups()
    letter = letters[0] if letters else ''

    return int(number), letter


def fix_broken_classes(broken_classes):
    if broken_classes and CoachType.UNKNOWN.value in broken_classes:
        broken_classes[CoachType.UNKNOWN.value] = [
            r for r in broken_classes[CoachType.UNKNOWN.value]
            if r not in (
                TariffError.TOO_CHEAP.value,
                TariffError.UNSUPPORTED_COACH_TYPE.value,
                TariffError.SERVICE_NOT_ALLOWED.value
            )
        ]
        if not broken_classes[CoachType.UNKNOWN.value]:
            del broken_classes[CoachType.UNKNOWN.value]
    return broken_classes
