# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from datetime import datetime, timedelta
from decimal import Decimal

from common.apps.train_order.enums import CoachType
from common.models.currency import Price
from common.utils.date import smart_localize, timedelta2minutes
from common.utils.title_generator import DASH
from travel.rasp.train_api.tariffs.train.base.availability_indication import AvailabilityIndication
from travel.rasp.train_api.tariffs.train.base.coach_category_traits import CoachCategoryTraits
from travel.rasp.train_api.tariffs.train.base.models import TrainSegment, TrainTariff, PlaceReservationType
from travel.rasp.train_api.tariffs.train.base.train_tariffs import build_train_tariffs_classes
from travel.rasp.train_api.tariffs.train.base.utils import (
    get_possible_numbers, fix_train_number, make_tariff_segment_key, fix_broken_classes
)
from travel.rasp.train_api.train_partners.im.base import IM_DATETIME_FORMAT, IM_BUS_TRANSPORT_TYPE

log = logging.getLogger(__name__)
log_data = logging.getLogger(__name__ + '.data')


IM_PLACE_RESERVATION_TYPES = {
    'Usual': PlaceReservationType.USUAL,
    'TwoPlacesAtOnce': PlaceReservationType.TWO_PLACES_AT_ONCE,
    'FourPlacesAtOnce': PlaceReservationType.FOUR_PLACES_AT_ONCE
}

IM_AVAILABILITY_INDICATION_TYPES = {
    'Available': AvailabilityIndication.AVAILABLE,
    'NotAvailableInWeb': AvailabilityIndication.NOT_AVAILABLE_IN_WEB,
    'FeatureNotAllowed': AvailabilityIndication.FEATURE_NOT_ALLOWED,
    'ServiceNotAllowed': AvailabilityIndication.SERVICE_NOT_ALLOWED,
    'CarrierNotAllowedForSales': AvailabilityIndication.CARRIER_NOT_ALLOWED_FOR_SALES,
    'OtherReasonOfInaccessibility': AvailabilityIndication.OTHER_REASON_OF_INACCESSIBILITY,
}


def filter_bus_segments(raw_segments):
    return (raw_segment for raw_segment in raw_segments
            if raw_segment.get('TransportType', '').strip().upper() != IM_BUS_TRANSPORT_TYPE.upper())


def build_train_segments(response_data, query):
    if not response_data:
        return []

    raw_segments_iter = filter_bus_segments(response_data['Trains'])
    segments_iter = (_build_train_segment(im_train, query) for im_train in raw_segments_iter)
    return [s for s in segments_iter if s]


def _get_im_time(datetime_string, tz):
    naive_datetime = datetime.strptime(datetime_string, IM_DATETIME_FORMAT)
    return smart_localize(naive_datetime, tz)


def _fill_inplace_segment_time_and_stations(train_segment, im_train, query):
    railway_departure = _get_im_time(im_train['DepartureDateTime'], query.departure_railway_tz)
    railway_arrival = _get_im_time(im_train['ArrivalDateTime'], query.arrival_railway_tz)

    _check_duration_and_timezones(im_train, query, railway_departure, railway_arrival)

    train_segment.station_from_express_code = im_train['OriginStationCode']
    train_segment.station_to_express_code = im_train['DestinationStationCode']

    train_segment.start_express_title_or_code = im_train['OriginName']
    train_segment.end_express_title_or_code = im_train['DestinationName']
    train_segment.railway_departure = railway_departure
    train_segment.railway_arrival = railway_arrival
    train_segment.departure = railway_departure.astimezone(query.departure_point.pytz)
    train_segment.arrival = railway_arrival.astimezone(query.arrival_point.pytz)
    train_segment.is_deluxe = 'ФИРМ' in (im_train['TrainDescription'] or '')


def _check_duration_and_timezones(im_train, query, railway_departure, railway_arrival):
    duration = timedelta(minutes=im_train['TripDuration'])
    if railway_arrival - railway_departure != duration:
        log.warning('Рассчитанное и указанное время прибытия отличаются %s %s', railway_departure, railway_arrival)

    railway_departure_from_local_dt = _get_im_time(im_train['LocalDepartureDateTime'], query.departure_point.pytz)
    if railway_departure != railway_departure_from_local_dt:
        log.error(
            'Для пункта %s расхождение временных зон составляет %s минут. Запрос %s',
            query.departure_point,
            timedelta2minutes(railway_departure - railway_departure_from_local_dt),
            query
        )

    railway_arrival_from_local_dt = _get_im_time(im_train['LocalArrivalDateTime'], query.arrival_point.pytz)
    if railway_arrival != railway_arrival_from_local_dt:
        log.error(
            'Для пункта %s расхождение временных зон составляет %s минут. Запрос %s',
            query.arrival_point,
            timedelta2minutes(railway_arrival - railway_arrival_from_local_dt),
            query
        )


def _build_train_segment(im_train, query):
    segment = TrainSegment()
    segment.raw_train_category = im_train['TrainDescription'].strip() if im_train['TrainDescription'] else None
    segment.raw_train_name = im_train['TrainName'].strip() if im_train['TrainName'] else None
    segment.original_number = fix_train_number(im_train['TrainNumber'])
    segment.number = fix_train_number(im_train['DisplayTrainNumber']) or segment.original_number
    segment.train_number_to_get_route = im_train['TrainNumberToGetRoute']
    segment.possible_numbers = get_possible_numbers(segment.original_number)
    segment.tariffs['electronic_ticket'] = im_train['HasElectronicRegistration']
    segment.can_supply_segments = True

    segment.ufs_title = ' {} '.format(DASH).join([im_train['OriginName']] + im_train['DestinationNames'])

    segment.coach_owners = im_train['Carriers']
    segment.has_dynamic_pricing = im_train['HasDynamicPricingCars']
    segment.two_storey = im_train['HasTwoStoreyCars']
    segment.is_suburban = im_train['IsSuburban']
    segment.provider = im_train['Provider']
    try:
        _fill_inplace_segment_time_and_stations(segment, im_train, query)
    except Exception:
        log.exception('Ошибка разбора дополнительной информации о поезде:\n %s', im_train)
        return

    segment.key = make_tariff_segment_key(segment)

    classes, broken_classes = _build_classes(im_train)
    broken_classes = fix_broken_classes(broken_classes)
    if not classes and not broken_classes:
        return

    segment.tariffs['classes'] = classes
    segment.tariffs['broken_classes'] = broken_classes

    return segment


def _parse_train_tariff(im_coach_group):
    coach_type = CoachType.safe_from_im_code(im_coach_group['CarType'])
    place_reservation_type = parse_place_reservation_types(im_coach_group)

    min_tariff = Decimal(im_coach_group['MinPrice'])
    max_tariff = Decimal(im_coach_group['MaxPrice'])
    service_tariff = Decimal(im_coach_group['ServiceCosts'][0])
    seats = im_coach_group['TotalPlaceQuantity']
    several_prices = max_tariff != min_tariff
    service_class = im_coach_group['ServiceClasses'][0] if im_coach_group.get('ServiceClasses') else None
    category_traits = map(CoachCategoryTraits.loads, im_coach_group['CarDescriptions'])

    return TrainTariff(
        coach_type=coach_type,
        ticket_price=Price(min_tariff, 'RUB'),
        service_price=Price(service_tariff, 'RUB'),
        several_prices=several_prices,
        seats=seats,
        lower_seats=im_coach_group['LowerPlaceQuantity'],
        upper_seats=im_coach_group['UpperPlaceQuantity'],
        lower_side_seats=im_coach_group['LowerSidePlaceQuantity'],
        upper_side_seats=im_coach_group['UpperSidePlaceQuantity'],
        max_seats_in_the_same_car=seats,
        place_reservation_type=place_reservation_type,
        is_transit_document_required=im_coach_group['IsTransitDocumentRequired'],
        availability_indication=IM_AVAILABILITY_INDICATION_TYPES.get(
            im_coach_group['AvailabilityIndication'], AvailabilityIndication.UNKNOWN
        ),
        service_class=service_class,
        category_traits=category_traits,
        has_non_refundable_tariff=im_coach_group['HasNonRefundableTariff']
    )


def parse_place_reservation_types(im_coach_group):
    im_place_reservation_type = im_coach_group['PlaceReservationTypes'][0]
    return IM_PLACE_RESERVATION_TYPES.get(im_place_reservation_type, PlaceReservationType.UNKNOWN)


def _build_classes(im_train):
    tariffs = [
        _parse_train_tariff(im_coach_group)
        for im_coach_group in im_train['CarGroups']
    ]

    return build_train_tariffs_classes(tariffs)
