# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import logging

from django.conf import settings
from django.core.serializers.json import DjangoJSONEncoder

from common.settings.configuration import Configuration
from common.settings.utils import define_setting

scarab_log = logging.getLogger('train_pricing')

define_setting('IM_TRAIN_PRICING_LOG_ENABLED', {
    Configuration.PRODUCTION: True
}, default=False)


def format_list(source_list):
    if not source_list:
        return None
    if not any(source_list):
        return None
    return ', '.join(source_list)


def format_dict(source_dict):
    return json.dumps(source_dict, ensure_ascii=False, cls=DjangoJSONEncoder)


def log_response(train_query, response_data):
    if not response_data:
        return
    if not settings.IM_TRAIN_PRICING_LOG_ENABLED:
        return
    import scarab.common.events as common
    import scarab.main as scarab
    ts = int(scarab.time.time() * 1000)
    query_params = {
        'q_origin': train_query.departure_point_code,
        'q_destination': train_query.arrival_point_code,
        'q_departure_date': train_query.departure_date.strftime('%Y-%m-%dT00:00:00'),
    }
    for im_train in response_data['Trains']:
        train_params = {
            't_train_name': im_train['TrainName'],
            't_train_number': im_train['TrainNumber'],
            't_train_number_to_get_route': im_train['TrainNumberToGetRoute'],
            't_display_train_number': im_train['DisplayTrainNumber'],
            't_origin_name': im_train['OriginName'],
            't_origin_station_code': im_train['OriginStationCode'],
            't_departure_date_time': im_train['DepartureDateTime'],
            't_local_departure_date_time': im_train['LocalDepartureDateTime'],
            't_departure_stop_time': im_train['DepartureStopTime'],
            't_destination_name': im_train['DestinationName'],
            't_destination_station_code': im_train['DestinationStationCode'],
            't_arrival_date_time': im_train['ArrivalDateTime'],
            't_local_arrival_date_time': im_train['LocalArrivalDateTime'],
            't_arrival_stop_time': im_train['ArrivalStopTime'],
            't_car_services': format_list(im_train['CarServices']),
            't_carriers': format_list(im_train['Carriers']),
            't_has_dynamic_pricing_cars': im_train['HasDynamicPricingCars'],
            't_has_electronic_registration': im_train['HasElectronicRegistration'],
            't_has_two_storey_cars': im_train['HasTwoStoreyCars'],
            't_id': im_train['Id'],
            't_initial_station_name': im_train['InitialStationName'],
            't_is_component': im_train['IsComponent'],
            't_is_sale_forbidden': im_train['IsSaleForbidden'],
            't_is_suburban': im_train['IsSuburban'],
            't_train_description': im_train['TrainDescription'],
            't_transport_type': im_train['TransportType'],
            't_departure_date_from_forming_station': im_train['DepartureDateFromFormingStation'],
            't_final_station_name': im_train['FinalStationName'],
            't_trip_distance': im_train['TripDistance'],
            't_trip_duration': im_train['TripDuration'],
            't_provider': im_train['Provider'],
            't_booking_system': im_train['BookingSystem'],
        }
        for im_car in im_train['CarGroups']:
            event_params = {
                'c_agent_fee_calculation': format_dict(im_car['AgentFeeCalculation']),
                'c_availability_indication': im_car['AvailabilityIndication'],
                'c_car_descriptions': format_list(im_car['CarDescriptions']),
                'c_car_type': im_car['CarType'],
                'c_car_type_name': im_car['CarTypeName'],
                'c_carriers': format_list(im_car['Carriers']),
                'c_client_fee_calculation': im_car['ClientFeeCalculation'],
                'c_discounts': format_dict(im_car['Discounts']),
                'c_has_electronic_registration': im_car['HasElectronicRegistration'],
                'c_has_gender_cabins': im_car['HasGenderCabins'],
                'c_has_non_refundable_tariff': im_car['HasNonRefundableTariff'],
                'c_has_place_numeration': im_car['HasPlaceNumeration'],
                'c_has_places_near_babies': im_car['HasPlacesNearBabies'],
                'c_has_places_near_pets': im_car['HasPlacesNearPets'],
                'c_has_places_near_playground': im_car['HasPlacesNearPlayground'],
                'c_info_request_schema': im_car['InfoRequestSchema'],
                'c_international_service_classes': format_list(im_car['InternationalServiceClasses']),
                'c_is_additional_meal_option_possible': im_car['IsAdditionalMealOptionPossible'],
                'c_is_bedding_selection_possible': im_car['IsBeddingSelectionPossible'],
                'c_is_interstate': im_car['IsInterstate'],
                'c_is_meal_option_possible': im_car['IsMealOptionPossible'],
                'c_is_on_request_meal_option_possible': im_car['IsOnRequestMealOptionPossible'],
                'c_is_sale_forbidden': im_car['IsSaleForbidden'],
                'c_is_three_hours_reservation_available': im_car['IsThreeHoursReservationAvailable'],
                'c_is_transit_document_required': im_car['IsTransitDocumentRequired'],
                'c_place_reservation_types': format_list(im_car['PlaceReservationTypes']),
                'c_service_classes': format_list(im_car['ServiceClasses']),
                'c_service_costs': float(im_car['ServiceCosts'][0]),
                'c_max_price': float(im_car['MaxPrice']),
                'c_min_price': float(im_car['MinPrice']),
                'c_total_place_quantity': im_car['TotalPlaceQuantity'],
                'c_place_quantity': im_car['PlaceQuantity'],
                'c_mixed_cabin_quantity': im_car['MixedCabinQuantity'],
                'c_upper_place_quantity': im_car['UpperPlaceQuantity'],
                'c_upper_side_place_quantity': im_car['UpperSidePlaceQuantity'],
                'c_lower_place_quantity': im_car['LowerPlaceQuantity'],
                'c_lower_side_place_quantity': im_car['LowerSidePlaceQuantity'],
                'c_male_place_quantity': im_car['MalePlaceQuantity'],
                'c_female_place_quantity': im_car['FemalePlaceQuantity'],
                'c_empty_cabin_quantity': im_car['EmptyCabinQuantity'],
            }
            event_params.update(**query_params)
            event_params.update(**train_params)
            event = scarab.create_event(
                common.EventType.TRAINS_IM_TRAIN_PRICING_EVENT,
                ts,
                common.Provider.TRAINS,
                2,
                scarab.CreationPolicy.CREATE_FAILURE_EVENT,
                **event_params)
            scarab_log.info(scarab.serialize_event_to_str(event))
