# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import copy
import re
from collections import namedtuple, defaultdict

from common.models.schedule import RThreadType
from common.models.transport import TransportType

TrainNumber = namedtuple('TrainNumber', ['digits', 'letters'])
train_number_re = re.compile(r'(\d+)(\D+)$')


def is_through_train(segment):
    return segment.thread and segment.thread.type_id == RThreadType.THROUGH_TRAIN_ID


def parse_train_number(train_number):
    match = train_number_re.match(train_number)
    if not match:
        return None
    return TrainNumber(*match.groups())


def fill_train_numbers(segments):
    for segment in segments:
        if segment.t_type.id == TransportType.TRAIN_ID:
            segment.train_number = parse_train_number(segment.number)


def choose_main_segment(segments):
    # Выбираем основную нитку, если она есть. Преимущество у быстрых поездов
    return min(segments, key=lambda x: (is_through_train(x), x.duration))


def create_meta_train(main_segment, segments):
    meta_train = copy.copy(main_segment)
    meta_train.sub_segments = segments
    all_letters = ''.join(sorted({letter for segment in segments for letter in segment.train_number.letters}))
    meta_train.number = '{}{}'.format(main_segment.train_number.digits, all_letters)
    meta_train.letters = {segment.train_number.letters for segment in segments}
    meta_train.letters.add(all_letters)
    meta_train.companies = {segment.company for segment in segments if segment.company}
    meta_train.min_arrival = min([segment.arrival for segment in segments])
    meta_train.max_arrival = max([segment.arrival for segment in segments])
    return meta_train


def make_meta_trains(segments):
    fill_train_numbers(segments)
    train_numbers = defaultdict(list)
    for segment in segments:
        if segment.t_type.id == TransportType.TRAIN_ID and segment.train_number:
            train_numbers[(segment.train_number.digits,
                          segment.departure)].append(segment)

    for v in train_numbers.values():
        if len(v) >= 2:
            main_segment = choose_main_segment(v)
            segments.append(create_meta_train(main_segment, v))
            for segment in v:
                segment.is_sub_segment = True

    return [segment for segment in segments if not getattr(segment, 'is_sub_segment', False)]
