# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import json
import logging

from common.dynamic_settings.default import conf
from common.models.geo import StationExpressAlias, StationCode, StationMajority, Station
from common.models.transport import TransportType
from common.utils.title_generator import build_default_title_common, TitleGenerator
from travel.rasp.train_api.pb_provider import PROTOBUF_DATA_PROVIDER

log = logging.getLogger(__name__)


def fill_segment_title_common(ufs_segments):
    names_or_codes = set()
    t_type = TransportType.objects.get(pk=TransportType.TRAIN_ID)

    for s in ufs_segments:
        if s.thread:
            s.title = s.thread.title
            s.title_common = s.thread.title_common

    segments_without_thread = [_s for _s in ufs_segments if not _s.thread]

    for s in segments_without_thread:
        names_or_codes.add(s.start_express_title_or_code)
        names_or_codes.add(s.end_express_title_or_code)

    if conf.TRAIN_BACKEND_USE_PROTOBUFS['alias']:
        alias_repo = PROTOBUF_DATA_PROVIDER.alias_repo
        station_id_to_alias = {}

        for name in names_or_codes:
            bin_name = name.encode('utf-8')  # Workaround: Прото ресурс возвращает бинарные строки
            proto = alias_repo.get(bin_name)
            if proto:
                station_id_to_alias[proto.StationId] = name

        stations = Station.objects.filter(id__in=station_id_to_alias).select_related('settlement')
        name_to_station = {station_id_to_alias[station.id]: station for station in stations}
    else:
        name_to_station = {
            ea.alias: ea.station
            for ea in StationExpressAlias.objects.filter(alias__in=names_or_codes).select_related('station__settlement')
        }

    code_to_station = {
        sc.code: sc.station
        for sc in StationCode.objects.filter(
            system__code='express',
            code__in=names_or_codes,
        ).select_related('station__settlement')
    }

    for ufs_segment in segments_without_thread:
        start_point = name_to_station.get(ufs_segment.start_express_title_or_code) or \
            code_to_station.get(ufs_segment.start_express_title_or_code)
        end_point = name_to_station.get(ufs_segment.end_express_title_or_code) or \
            code_to_station.get(ufs_segment.end_express_title_or_code)

        if not start_point:
            log.warning('Не нашли начальную Экспресс станцию по названию или коду: %s',
                        ufs_segment.start_express_title_or_code)
        if not end_point:
            log.warning('Не нашли конечную Экспресс станцию по названию или коду: %s',
                        ufs_segment.end_express_title_or_code)

        if not (start_point and end_point):
            ufs_segment.can_supply_segments = False
            continue

        if isinstance(start_point, Station) and start_point.majority_id == StationMajority.EXPRESS_FAKE_ID:
            start_point = start_point.settlement

        if isinstance(end_point, Station) and end_point.majority_id == StationMajority.EXPRESS_FAKE_ID:
            end_point = end_point.settlement

        ufs_segment.title_common = build_default_title_common(t_type, [start_point, end_point])
        ufs_segment.title = TitleGenerator.L_title(
            json.loads(ufs_segment.title_common), lang='ru', popular=False
        )

    return ufs_segments
