# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

"""
Формирование сегментов с ценами и/или причинами отсутствия цен
--------------------------------------------------------------
Партнер не отдает в поиске сегменты, для которых билеты уже проданы. Поэтому используем данные из расписания
и глубину продаж для определения сегментов с проданными билетами.

1. Нужны станции отправления и прибытия пассажира
- заполняем из запроса, если можем, потому что не нужно доставать из базы
- заполняем из ответа партнера (раньше поискового сегмента, чтобы залогировать ошибку)
- заполняем из поискового сегмента (для поискового сегмента, уже вытащены из базы)

2. Для построения title и title_common нужны начальная и конечная станция поезда с городами. Нужно только для
сегментов без нитки, потому что можно взять title и title_common из нитки.

3. Для построения ссылки на выбор мест нужны коды страны начальной и конечной станции поезда.

Заполняем segment.first_country_code и segment.last_country_code, для построения правилной ссыылки,
потому что и для сегментов без цен можем заполнить данный параметр.
Заполняем segment.thread.first_country_code и segment.thread.last_country_code, потому что их ожидает увидеть фронт.

4. Так же для station_from и station_to раньше заполняли страну -- это скорее всего не нужно, не будем в первом варианте
так делать. Есть коммит https://github.yandex-team.ru/rasp-junk/train_api/commit/42d3ac4133722a2bb9e15476339861dfa0f7db49
но без таска... похоже это для поиска на все дни https://github.yandex-team.ru/rasp-junk/morda_backend/pull/447

5. Если не заполнили station_from или station_to, то логируем и пропускаем такой сегмент.

6. Если нет start_station или end_station, что возможно только для пополнений, то ставим флаг не пополнять
и оставляем начальную и конечную страны незаполненными, что приведет к ссылке на УФС.

Таблица заполнения нужных данных:
+---------------+------------------------------------------+----------------------------+--------------+
|               | Price + Thread                           | Price                      | Thread       |
+---------------+------------------------------------------+----------------------------+--------------+
| station_from  | - из запроса                             | - из запроса               | - из запроса |
| station_to    | - из ответа партнера, чтобы залогировать | - из ответа партнера       | - из нитки   |
|               | - из нитки                               |                            |              |
+---------------+------------------------------------------+----------------------------+--------------+
| start_station | - из ответа партнера, чтобы залогировать | из ответа партнера         | из нитки     |
| end_station   | - из нитки                               |                            |              |
+---------------+------------------------------------------+----------------------------+--------------+
| start_country | из базы                                  | из базы                    | из базы      |
| end_country   |                                          |                            |              |
+---------------+------------------------------------------+----------------------------+--------------+
| title         | из нитки                                 | start_station, end_station | из нитки     |
| title_common  |                                          | + settlement               |              |
+---------------+------------------------------------------+----------------------------+--------------+
"""

import json
import logging

from django.utils.lru_cache import lru_cache

from common.apps.train.tariff_error import TariffError
from common.apps.train_order.enums import CoachType
from common.dynamic_settings.default import conf
from common.models.geo import StationMajority, Station, StationExpressAlias
from common.models.schedule import RThreadType
from common.models.transport import TransportType
from common.models_utils import fetch_related
from travel.rasp.library.python.common23.date import environment
from common.utils.title_generator import build_default_title_common, TitleGenerator
from travel.rasp.train_api.pb_provider import PROTOBUF_DATA_PROVIDER
from travel.rasp.train_api.tariffs.train.base.models import TrainSegment
from travel.rasp.train_api.tariffs.train.base.utils import make_tariff_segment_key_exact
from travel.rasp.train_api.tariffs.train.segment_builder.helpers.black_list import redirect_blacklisted_trains_to_ufs
from travel.rasp.train_api.tariffs.train.segment_builder.helpers.sales_depth import sales_depth_manager
from travel.rasp.train_api.train_partners.im.base import PROVIDER_P2

log = logging.getLogger(__name__)

SOLD_OUT_BROKEN_CLASSES = {CoachType.UNKNOWN.value: [TariffError.SOLD_OUT.value]}


def build_price_segments_with_reason_for_missing_prices(segments, search_segments, train_query):
    # type: (List[TrainSegment], TrainTariffsQuery) -> List[TrainSegment]
    segments = fill_station_from_and_station_to_from_partner_response(segments, train_query)
    segments = fill_start_and_end_stations_from_partner_response(segments)

    full_segments, segments_without_thread, no_price_segments = classify_price_segments(segments, search_segments)

    segments = (
        fill_full_segments(full_segments)
        + fill_segments_without_thread(segments_without_thread)
        + keep_only_sold_out_segments(fill_no_price_segments(no_price_segments), train_query)
    )

    segments = filter_segments_without_station_from_or_station_to(segments, train_query)
    segments = fill_first_and_last_country_codes(segments)
    segments = redirect_blacklisted_trains_to_ufs(segments)
    return segments


def fill_station_from_and_station_to_from_partner_response(segments, train_query):
    @lru_cache()
    def get_express_station(express_code):
        return Station.get_by_code('express', express_code)

    express_point_from = get_express_station(train_query.departure_point_code)
    express_point_to = get_express_station(train_query.arrival_point_code)

    can_fill_stations_from_request = (
        express_point_from.majority_id != StationMajority.EXPRESS_FAKE_ID
        and express_point_to.majority_id != StationMajority.EXPRESS_FAKE_ID
    )

    for segment in segments:
        if can_fill_stations_from_request:
            segment.station_from = express_point_from
            segment.station_to = express_point_to
        else:
            try:
                segment.station_from = get_express_station(segment.station_from_express_code)
            except Station.DoesNotExist:
                log.error('Не нашли станцию по экспресс коду %s', segment.station_from_express_code)

            try:
                segment.station_to = get_express_station(segment.station_to_express_code)
            except Station.DoesNotExist:
                log.error('Не нашли станцию по экспресс коду %s', segment.station_to_express_code)

    return segments


def fill_start_and_end_stations_from_partner_response(segments):
    names = []
    for segment in segments:
        names.append(segment.start_express_title_or_code)  # легаси, для ИМ это title
        names.append(segment.end_express_title_or_code)

    if conf.TRAIN_BACKEND_USE_PROTOBUFS['alias']:
        alias_repo = PROTOBUF_DATA_PROVIDER.alias_repo
        station_id_to_alias = {}

        for name in names:
            bin_name = name.encode('utf-8')  # Workaround: Прото ресурс возвращает бинарные строки
            proto = alias_repo.get(bin_name)
            if proto:
                station_id_to_alias[proto.StationId] = name

        stations = Station.objects.filter(id__in=station_id_to_alias)
        name_to_station = {station_id_to_alias[station.id]: station for station in stations}
    else:
        name_to_station = {
            ea.alias: ea.station
            for ea in StationExpressAlias.objects.filter(alias__in=names).select_related('station')
        }

    for segment in segments:
        segment.start_station = name_to_station.get(segment.start_express_title_or_code)
        if not segment.start_station:
            _log_error_fund_station_by_name(segment.start_express_title_or_code)
        segment.end_station = name_to_station.get(segment.end_express_title_or_code)
        if not segment.end_station:
            _log_error_fund_station_by_name(segment.end_express_title_or_code)

    return segments


def _log_error_fund_station_by_name(station_name):
    log.warning('StationExpressAlias: не нашли станцию по названию %s', station_name)


def classify_price_segments(segments, search_segments):
    for segment in segments:
        if (segment.provider == PROVIDER_P2 and
                conf.TRAIN_PURCHASE_ENABLE_CPPK_CONDITIONS and
                segment.original_number in conf.TRAIN_PURCHASE_P2_TRAIN_NUMBERS_MAP):
            for number in conf.TRAIN_PURCHASE_P2_TRAIN_NUMBERS_MAP[segment.original_number]:
                key = make_tariff_segment_key_exact(number, segment.departure)
                matching_search_segments = [s for s in search_segments if key in s.train_keys]
                if matching_search_segments:
                    segment.key = key
                    break
        else:
            matching_search_segments = [s for s in search_segments if segment.key in s.train_keys]
        if not matching_search_segments:
            search_segment = None
        elif len(matching_search_segments) == 1:
            search_segment = matching_search_segments[0]
        else:
            search_segment = min(matching_search_segments, key=lambda t: (t.thread.type_id == RThreadType.THROUGH_TRAIN_ID,
                                                                          t.station_from.majority_id, t.station_to.majority_id))
        if search_segment:
            segment.thread = search_segment.thread
            segment.search_segment = search_segment
            search_segment.used = True

    full_segments = [s for s in segments if s.thread]
    segments_without_thread = [s for s in segments if not s.thread]
    unused_search_segments = [s for s in search_segments if not hasattr(s, 'used')]
    no_price_segments = sum((make_no_price_segments_from_search_segment(s) for s in unused_search_segments), [])
    return full_segments, segments_without_thread, no_price_segments


def keep_only_sold_out_segments(no_price_segments, train_query):
    days_from_today = (
        train_query.departure_date
        - environment.now_aware().astimezone(train_query.departure_point.pytz).date()
    ).days

    sold_out_segments = []
    for segment in no_price_segments:
        sales_depth = sales_depth_manager.get_sales_depth(segment)
        if sales_depth and days_from_today < sales_depth:
            sold_out_segments.append(segment)

    return sold_out_segments


def make_no_price_segments_from_search_segment(search_segment):
    # type: (List[RThreadSegment]) -> List[TrainSegment]
    segments = []
    for key in search_segment.train_keys:
        segment = TrainSegment()
        segment.can_supply_segments = False  # не надо использовать поисковые сегменты без цены, как сегменты пополнения
        segment.key = key
        segment.original_number = search_segment.thread.number

        segment.number = search_segment.number
        rtstation_from = getattr(search_segment, 'rtstation_from', None)
        if rtstation_from is not None and rtstation_from.train_number:
            segment.number = rtstation_from.train_number

        segment.t_model = search_segment.thread.t_model
        segment.departure = search_segment.departure
        segment.arrival = search_segment.arrival
        segment.is_deluxe = search_segment.thread.is_deluxe
        segment.is_suburban = search_segment.thread.t_type_id == TransportType.SUBURBAN_ID
        segment.search_segment = search_segment
        segment.thread = search_segment.thread
        segment.coach_owners = build_coach_owners(search_segment.thread)
        segment.tariffs = {
            'classes': {},
            'broken_classes': SOLD_OUT_BROKEN_CLASSES
        }
        segments.append(segment)
    return segments


def fill_full_segments(segments):
    for segment in segments:
        segment.station_from = segment.station_from or segment.search_segment.station_from
        segment.station_to = segment.station_to or segment.search_segment.station_to

        segment.start_station = segment.start_station or segment.search_segment.start_station
        segment.end_station = segment.end_station or segment.search_segment.end_station

        segment.title = segment.thread.title
        segment.title_common = segment.thread.title_common

        segment.coach_owners = build_coach_owners(segment.thread) or segment.coach_owners

    return segments


def build_coach_owners(thread):
    if not thread:
        return ()

    company_title = None
    if thread.company:
        company_title = thread.company.short_title or thread.company.title
    return (company_title,) if company_title else ()


def fill_segments_without_thread(segments):
    t_type = TransportType.objects.get(pk=TransportType.TRAIN_ID)
    fetch_related([s.start_station for s in segments if s.start_station] + [s.end_station for s in segments if s.end_station],
                  'settlement', model=Station)
    for segment in segments:
        if not segment.start_station or not segment.end_station:
            segment.can_supply_segments = False
            continue

        segment.title_common = build_default_title_common(t_type, [segment.start_station, segment.end_station])
        segment.title = TitleGenerator.L_title(
            json.loads(segment.title_common), lang='ru', popular=False
        )

    return segments


def fill_no_price_segments(segments):
    for segment in segments:
        segment.station_from = segment.search_segment.station_from
        segment.station_to = segment.search_segment.station_to

        segment.start_station = segment.search_segment.start_station
        segment.end_station = segment.search_segment.end_station

        segment.title = segment.thread.title
        segment.title_common = segment.thread.title_common

    return segments


def filter_segments_without_station_from_or_station_to(segments, train_query):
    for segment in segments:
        if not segment.station_from or not segment.station_to:
            log.error('Не смогли восстановить станции отправления и(или) прибытия не показываем цены для %s, %s',
                      segment.original_number, train_query)

    return [s for s in segments if s.station_from and s.station_to]


def fill_first_and_last_country_codes(segments):
    fetch_related([s.start_station for s in segments if s.start_station] + [s.end_station for s in segments if s.end_station],
                  'country', model=Station)
    for segment in segments:
        first_country_code = segment.start_station and segment.start_station.country and segment.start_station.country.code
        last_country_code = segment.end_station and segment.end_station.country and segment.end_station.country.code
        segment.first_country_code = first_country_code
        segment.last_country_code = last_country_code
        if segment.thread:
            # Нужно заполнить атрибуты, даже если это будет None
            segment.thread.first_country_code = first_country_code
            segment.thread.last_country_code = last_country_code
        if not first_country_code or not last_country_code:
            segment.can_supply_segments = False
    return segments
