# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from rest_framework import status
from rest_framework.decorators import api_view
from rest_framework.response import Response

from common.data_api.min_prices.api import min_price_storage
from common.models.transport import TransportType
from route_search.shortcuts import search_routes, find
from travel.rasp.train_api.tariffs.train.base import price_logging
from travel.rasp.train_api.tariffs.train.base.query import get_train_segments
from travel.rasp.train_api.tariffs.train.base.serialization import (
    TrainQuerySchema, TrainResponseSchema,
    TrainEarliestDateQuerySchema, TrainEarliestDateResponseSchema,
    MinTariffsQuerySchema, MinTariffsResponseSchema,
    MinTariffsShortQuerySchema, MinTariffsShortResponseSchema
)
from travel.rasp.train_api.tariffs.train.segment_builder.helpers.merge_trains import make_meta_trains
from travel.rasp.train_api.tariffs.train.segment_builder.helpers.sales_depth import sales_depth_manager
from travel.rasp.train_api.tariffs.train.segment_builder.search_segments import (
    fill_start_and_end_stations_from_thread
)
from travel.rasp.train_api.tariffs.train.wizard.service import get_wizard_prices
from travel.rasp.train_api.train_purchase.core.enums import TrainPartner


@api_view(['GET'])
def train_tariffs(request, poll=False):
    """
    Поиск тарифов.
    Пример запроса Питер - Москва:
    /ru/api/segments/train-tariffs/?pointFrom=c2&pointTo=c213&date=2016-05-10&national_version=ru
    Результат можно получить по адресу:
    /ru/api/segments/train-tariffs/poll/?pointFrom=c2&pointTo=c213&date=2016-05-10&national_version=ru
    Должен работать кеш
    ---
    parameters:
      - in: query
        name: query_params
        schema:
            $ref: 'TrainQuerySchema'
    responses:
        200:
            description: поезда с тарифами для выдачи, либо errors
            schema:
                $ref: 'TrainResponseSchema'
    """
    query_params = TrainQuerySchema.prepare_query_from_headers(request.META, request.GET)
    query, errors = TrainQuerySchema().load(query_params)
    if errors:
        return Response({
            'errors': errors,
        }, status=status.HTTP_400_BAD_REQUEST)

    result = get_train_segments(
        query['point_from'], query['point_to'],
        query['start_time'], query['end_time'],
        include_price_fee=query['include_price_fee'],
        national_version=query['national_version'],
        can_ask_partner=TrainPartner.IM.enabled and not poll,
        yandex_uid=query['yandex_uid'] if query['price_exp_id'] else None,
        force_ufs_order=query['force_ufs_order'],
        allow_international_routes=query['allow_international_routes'],
        icookie=query['icookie'],
        bandit_type=query['bandit_type'],
        asker=query['asker'],
        ytp_referer=query['ytp_referer'],
        train_api_use_worker=query['train_api_use_worker'],
        original_query=query,
    )

    for segment in result.segments:
        price_logging.log_segment(segment, query)

    return Response(TrainResponseSchema(strict=True, context={'dump_country_for_station': True}).dump(result).data)


@api_view(['GET'])
def min_tariffs(request):
    """
    ---
    parameters:
      - in: query
        name: query_params
        schema:
            $ref: 'MinTariffsQuerySchema'
    responses:
        200:
            description: минимальные цены из колдунщика, либо errors
            schema:
                $ref: 'MinTariffsResponseSchema'
    """
    query, errors = MinTariffsQuerySchema().load(request.GET)
    if errors:
        return Response({
            'errors': errors,
        }, status=status.HTTP_400_BAD_REQUEST)

    result = get_wizard_prices(query)
    data = MinTariffsResponseSchema.fast_dump({'trains': result})
    return Response(data)


@api_view(['GET'])
def min_tariffs_short(request):
    """
    ---
    parameters:
      - in: query
        name: query_params
        schema:
            $ref: 'MinTariffsShortQuerySchema'
    responses:
        200:
            description: минимальные цены из монги, либо errors
            schema:
                $ref: 'MinTariffsShortResponseSchema'
    """
    query, errors = MinTariffsShortQuerySchema().load(request.GET)
    if errors:
        return Response({
            'errors': errors,
        }, status=status.HTTP_400_BAD_REQUEST)

    t_type = TransportType.objects.get(pk=TransportType.TRAIN_ID)
    min_prices = min_price_storage.get_min_prices(query['point_from'], query['point_to'], t_codes=[t_type.code])

    result = {}
    for item in min_prices:
        result_train = _get_or_create_min_tariffs_train(result, item['route_uid'])
        result_train['classes'][item['class']] = {
            'price': {
                'currency': 'RUB',
                'value': item['price'],
            }
        }

    if query.get('with_sales_depth'):
        search_segments, _, _ = search_routes(
            query['point_from'],
            query['point_to'],
            departure_date=None,
            transport_types=[t_type],
            exact_date=True,
            expanded_day=False,
            check_date=None,
            include_interval=False,
            add_z_tablos=False,
            max_count=None,
            threads_filter=None,
            prepared_threads=None
        )
        search_segments = make_meta_trains(search_segments)
        search_segments = fill_start_and_end_stations_from_thread(search_segments)
        for segment in search_segments:
            sales_depth = sales_depth_manager.get_sales_depth(segment)
            result_train = _get_or_create_min_tariffs_train(result, segment.number)
            result_train['sales_depth'] = sales_depth

    result = result.values()
    data = MinTariffsShortResponseSchema().dump({'tariffs': result}).data
    return Response(data)


@api_view(['GET'])
def train_tariffs_earliest_routed_date(request):
    """
    Поиск ближайшей даты, на которую есть рейсы.
    Пример запроса Питер - Москва:
    /ru/api/segments/train-tariffs/earliest_date/?pointFrom=c2&pointTo=c213
    ---
    parameters:
      - in: query
        name: query_params
        schema:
            $ref: 'TrainEarliestDateQuerySchema'
    responses:
        200:
            description: ближайшая дата, на которую есть рейсы, либо errors
            schema:
                $ref: 'TrainEarliestDateResponseSchema'
    """
    query_params = TrainEarliestDateQuerySchema.prepare_query_from_headers(request.META, request.GET)
    query, errors = TrainEarliestDateQuerySchema().load(query_params)
    if errors:
        return Response({
            'errors': errors,
        }, status=status.HTTP_400_BAD_REQUEST)

    search_segments = find(
        query['point_from'],
        query['point_to'],
        date_from=query.get('minimum_date'),
        t_type=TransportType.objects.get(pk=TransportType.TRAIN_ID),
    )

    earliest_date = None
    query_date = query.get('minimum_date')
    for s in search_segments:
        segment_date = s.departure.date()
        if earliest_date is None or earliest_date > segment_date:
            earliest_date = segment_date
        if earliest_date == query_date:
            break

    if earliest_date is None:
        return Response({
            'errors': 'Did not found any segments',
        }, status=status.HTTP_200_OK)

    result = {'date': earliest_date}

    return Response(TrainEarliestDateResponseSchema(strict=True).dump(result).data)


def _get_or_create_min_tariffs_train(result_dict, train_number):
    if train_number not in result_dict:
        result_dict[train_number] = {
            'key': train_number,
            'classes': {}
        }
    return result_dict[train_number]
