# coding: utf-8

from __future__ import unicode_literals, absolute_import, division, print_function

import base64
from logging import getLogger

from common.utils.date import naive_to_timestamp, UTC_TZ
from travel.rasp.library.python.common23.date.environment import now_utc
from travel.rasp.train_bandit_api.proto import api_pb2

bandit_train_details_logger = getLogger('bandit_train_details_logger')
log = getLogger(__name__)


def get_geo_id_from_station(station):
    settlement = station.settlement
    if settlement:
        return settlement._geo_id

    region = station.region
    if region:
        return region._geo_id

    country = station.country
    if country:
        return country._geo_id

    return None


def to_timestamp(dt):
    return naive_to_timestamp(dt.astimezone(UTC_TZ).replace(tzinfo=None))


def bandit_log_train_tariffs(segment, query):
    for class_name, tariff in segment.tariffs['classes'].items():
        try:
            data = {
                'timestamp': naive_to_timestamp(now_utc()),
                'test_buckets': query.get('test_buckets'),
                'req_id': query.get('req_id'),
                'icookie': query.get('icookie'),
                'yandex_uid': query.get('yandex_uid'),
                'user_device': query.get('user_device'),
                'point_from': segment.station_from.point_key,
                'point_to': segment.station_to.point_key,
                'departure_dt': to_timestamp(segment.departure),
                'arrival_dt': to_timestamp(segment.arrival),
                'station_from_geo_id': get_geo_id_from_station(segment.station_from),
                'station_to_geo_id': get_geo_id_from_station(segment.station_to),
                'train_type': segment.raw_train_name,
                'car_type': class_name,
                'service_class': tariff.service_class,
                'price': float(tariff.price.value),
                'fee': float(tariff.fee_percent),
                'is_bandit_fee_applied': tariff.is_bandit_fee_applied,
                'event_type': 'train-tariffs',
                'bandit_type': tariff.bandit_type,
                'bandit_version': tariff.bandit_version,
            }
            bandit_train_details_logger.info(msg=None, extra=data)
        except Exception:
            log.exception('Cannot log segment %s. Skipping.', segment.number)


def bandit_log_train_details(train_details, query):
    for coach in train_details.coaches:
        try:
            sorted_by_price_places = sorted(coach.places, key=lambda place: place.adult_tariff.value)
            data = {
                'timestamp': naive_to_timestamp(now_utc()),
                'test_buckets': query.get('test_buckets'),
                'req_id': query.get('req_id'),
                'icookie': query.get('icookie'),
                'yandex_uid': query.get('yandex_uid'),
                'user_device': query.get('user_device'),
                'point_from': train_details.station_from.point_key,
                'point_to': train_details.station_to.point_key,
                'departure_dt': to_timestamp(train_details.departure),
                'arrival_dt': to_timestamp(train_details.arrival),
                'station_from_geo_id': get_geo_id_from_station(train_details.station_from),
                'station_to_geo_id': get_geo_id_from_station(train_details.station_to),
                'train_type': train_details.raw_train_name,
                'car_type': coach.type,
                'service_class': coach.service_class_code,
                'price': None,
                'fee': None,
                'is_bandit_fee_applied': None,
                'event_type': 'train-details',
                'bandit_type': None,
                'bandit_version': None,
            }
            if sorted_by_price_places:
                cheapest_place = sorted_by_price_places[0]
                data.update({
                    'price': float(cheapest_place.adult_tariff.value),
                    'fee': float(cheapest_place.yandex_fee_percent),
                    'is_bandit_fee_applied': cheapest_place.is_bandit_fee_applied,
                    'bandit_type': cheapest_place.bandit_type,
                    'bandit_version': cheapest_place.bandit_version,
                })

            bandit_train_details_logger.info(msg=None, extra=data)
        except Exception:
            log.exception('Cannot log train details %s. Skipping.', train_details.start_number)


def bandit_log_passenger_details(data):
    fee_calculation_token_b64 = data.get('fee_calculation_token')
    data = {
        'timestamp': naive_to_timestamp(now_utc()),
        'test_buckets': data.get('test_buckets'),
        'req_id': data.get('req_id'),
        'icookie': data.get('icookie'),
        'yandex_uid': data.get('yandex_uid'),
        'user_device': data.get('user_device'),
        'point_from': data.get('station_from').point_key,
        'point_to': data.get('station_to').point_key,
        'departure_dt': to_timestamp(data.get('departure')),
        'arrival_dt': to_timestamp(data.get('arrival')),
        'station_from_geo_id': get_geo_id_from_station(data.get('station_from')),
        'station_to_geo_id': get_geo_id_from_station(data.get('station_to')),
        'train_type': data.get('train_type'),
        'car_type': data.get('car_type'),
        'service_class': data.get('service_class'),
        'price': float(data.get('price')),
        'fee': None,
        'is_bandit_fee_applied': None,
        'event_type': data.get('event_type') or 'passenger-details',
        'bandit_type': None,
        'bandit_version': None,
    }
    if fee_calculation_token_b64:
        fee_calculation_token = api_pb2.TFeeCalculationToken()
        fee_calculation_token.ParseFromString(base64.b64decode(fee_calculation_token_b64))
        data['fee'] = float(fee_calculation_token.Permille) / 1000.0
        data['is_bandit_fee_applied'] = fee_calculation_token.IsBanditFeeApplied
        data['bandit_type'] = fee_calculation_token.ActualBanditType
        data['bandit_version'] = fee_calculation_token.ActualBanditVersion
        data['icookie'] = fee_calculation_token.Context.ICookie
        data['point_from'] = fee_calculation_token.Context.PointFrom
        data['point_to'] = fee_calculation_token.Context.PointTo
        data['arrival_dt'] = fee_calculation_token.Context.Arrival.seconds
        data['departure_dt'] = fee_calculation_token.Context.Departure.seconds
        data['train_type'] = fee_calculation_token.Context.TrainType
        data['car_type'] = fee_calculation_token.Context.CarType
    bandit_train_details_logger.info(msg=None, extra=data)
