# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from collections import namedtuple
from decimal import Decimal

import six
from enum import IntEnum

from common.utils.yasmutil import MeasurableDecorator, Metric

Tax = namedtuple('Tax', ['rate', 'amount'])


def create_tax(rate, amount):
    if rate is None or amount is None:
        return None
    return Tax(rate=Decimal(rate), amount=Decimal(amount))


@six.python_2_unicode_compatible
class ApiError(Exception):
    code = None
    message = None

    def __init__(self, code=None, message=None, data=None):
        """
        :param data: Эти данные могут уйти на фронт, поэтому сюда нельзя писать ничего важного
        """
        self.code = code or self.code
        self.message = message or self.message
        self.data = data

        # Чтобы нормально pickle'лось в кеш
        self.args = code, message, data

    def get_data(self):
        return {
            'message': self.message,
            'data': self.data
        }

    def get_user_message(self):
        raise NotImplementedError

    def __eq__(self, other):
        return (self.code == other.code and
                self.message == other.message and
                self.data == other.data)

    def __str__(self):
        return '<{self.__class__.__name__}: code="{self.code}" message="{self.message}">'.format(self=self)


class PartnerError(ApiError):
    def __init__(self, code, message, data=None):
        self.code = code
        self.message = message
        self.data = data

    def is_communication_error(self):
        return False

    def is_retry_allowed(self):
        return False

    def is_refund_not_found_error(self):
        raise NotImplementedError

    def is_update_from_express_error(self):
        return False

    def is_non_cacheable_error(self):
        return False

    def is_empty_result_error(self):
        return False

    def __reduce__(self):
        return self.__class__, (self.code, self.message, self.data)


class RzhdStatus(IntEnum):
    """ Возможные значения:
        0: 'Без электронной регистрации',
        1: 'Электронная регистрация',
        2: 'Оплата не подтверждена',
        3: 'Аннулирован',
        4: 'Возвращен',
        5: 'Возвращены места',
        6: 'Выдан посадочный купон (проездной документ) на бланке строгой отчетности',
        7: 'Отложенная отплата (статус возвращается после создания бронирования)',
        8: 'Выполнено прерывание поездки',
        9: 'Выполнено прерывание с возобновлением поездки',
    """
    NO_REMOTE_CHECK_IN = 0
    REMOTE_CHECK_IN = 1
    PAYMENT_NOT_CONFIRMED = 2
    CANCELLED = 3
    REFUNDED = 4
    PLACES_REFUNDED = 5
    STRICT_BOARDING_PASS = 6
    RESERVATION = 7
    INTERRUPTED = 8
    RESUMED = 9
    UNKNOWN = 20

    def is_refundable(self):
        return self not in (
            RzhdStatus.CANCELLED, RzhdStatus.REFUNDED, RzhdStatus.PLACES_REFUNDED, RzhdStatus.STRICT_BOARDING_PASS,
            RzhdStatus.INTERRUPTED, RzhdStatus.RESUMED,
        )

    def is_possible_to_change_er_status(self):
        return self in (RzhdStatus.NO_REMOTE_CHECK_IN, RzhdStatus.REMOTE_CHECK_IN)


class AbstractPartnerMeasurable(MeasurableDecorator):
    def _handle_error(self, exc):
        result = super(AbstractPartnerMeasurable, self)._handle_error(exc)
        if hasattr(exc, 'is_communication_error'):
            error_type = 'communication_errors_cnt' if exc.is_communication_error() else 'other_errors_cnt'
            result.extend([
                Metric(self._name('errors_cnt'), 1, 'ammm'),
                Metric(self._name(error_type), 1, 'ammm'),
                Metric('errors_cnt', 1, 'ammm'),
                Metric(error_type, 1, 'ammm')
            ])
        return result
