# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import contextlib

import requests

from common.utils.namedtuple import namedtuple_with_defaults
from travel.rasp.train_api.train_partners.base import PartnerError
from travel.rasp.train_api.train_partners.base.get_order_info import get_order_info_by_reference_id
from travel.rasp.train_api.train_partners.base.refund import (
    PartnerRefundFailed, PartnerRefundUnknown, PartnerRefundNotFound, RefundResult
)
from travel.rasp.train_api.train_purchase.core.enums import TrainPartner, OperationStatus


InsuranceRefundResult = namedtuple_with_defaults(
    'InsuranceRefundResult',
    ('amount', 'buy_operation_id', 'refund_operation_id', 'order_customer_id'),
)


@contextlib.contextmanager
def check_refund_error():
    try:
        yield
    except PartnerError as error:
        if error.is_communication_error():
            raise PartnerRefundUnknown
        if error.is_refund_not_found_error():
            raise PartnerRefundNotFound

        raise PartnerRefundFailed
    except requests.RequestException:
        raise PartnerRefundUnknown


def insurance_refund(order, insurance_id, reference_id):
    from travel.rasp.train_api.train_partners import get_partner_api

    if order.partner != TrainPartner.IM:
        raise NotImplementedError('checkout не реализован для {}'.format(order.partner))
    with check_refund_error():
        return RefundResult(
            insurance_refund=get_partner_api(order.partner).insurance_refund(order, insurance_id, reference_id)
        )


def check_insurance_refund(order, reference_id):
    with check_refund_error():
        order_info = get_order_info_by_reference_id(order, reference_id)

    if order_info.status == OperationStatus.FAILED:
        raise PartnerRefundFailed

    if order_info.status == OperationStatus.IN_PROCESS:
        raise PartnerRefundUnknown

    insurance_info = None
    for i in order_info.insurances:
        if i.reference_id == reference_id:
            insurance_info = i
            break

    if not insurance_info:
        raise PartnerRefundNotFound

    return RefundResult(insurance_refund=InsuranceRefundResult(
        amount=insurance_info.refund_amount,
        buy_operation_id=insurance_info.operation_id,
        refund_operation_id=insurance_info.refund_operation_id,
        order_customer_id=insurance_info.customer_id,
    ))
