# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import contextlib

import requests

from common.utils.namedtuple import namedtuple_with_defaults
from travel.rasp.train_api.train_partners.base import PartnerError, RzhdStatus
from travel.rasp.train_api.train_partners.base.get_order_info import get_order_info_by_reference_id
from travel.rasp.train_api.train_purchase.core.enums import OperationStatus

TicketRefundResult = namedtuple_with_defaults(
    'TicketRefundResult',
    ('amount', 'refund_blank_id', 'refund_operation_id', 'rzhd_status'),
    defaults={'rzhd_status': RzhdStatus.REFUNDED}
)


class RefundResult(object):
    def __init__(self, refund_by_blank_id={}, insurance_refund=None):
        self.refund_by_blank_id = refund_by_blank_id
        self.insurance_refund = insurance_refund


class PartnerRefundException(Exception):
    pass


class PartnerRefundFailed(PartnerRefundException):
    pass


class PartnerRefundNotFound(PartnerRefundException):
    pass


class PartnerRefundUnknown(PartnerRefundException):
    pass


@contextlib.contextmanager
def check_refund_error():
    try:
        yield
    except PartnerError as error:
        if error.is_communication_error():
            raise PartnerRefundUnknown
        if error.is_refund_not_found_error():
            raise PartnerRefundNotFound

        raise PartnerRefundFailed
    except requests.RequestException:
        raise PartnerRefundUnknown


def make_refund(order, blank_id, doc_id, reference_id):
    from travel.rasp.train_api.train_partners import get_partner_api

    return get_partner_api(order.partner).make_refund(order, blank_id, doc_id, reference_id)


def check_refund(order, reference_id):
    with check_refund_error():
        order_info = get_order_info_by_reference_id(order, reference_id)

    if order_info.status == OperationStatus.FAILED:
        raise PartnerRefundFailed

    if order_info.status == OperationStatus.IN_PROCESS:
        raise PartnerRefundUnknown

    return _build_result_from_order_info(order_info)


def _build_result_from_order_info(order_info):
    refund_by_blank_id = {}
    for ticket_info in order_info.tickets:
        if not ticket_info.refund_blank_id:
            continue
        refund_by_blank_id[ticket_info.blank_id] = TicketRefundResult(
            amount=ticket_info.refund_amount,
            refund_blank_id=ticket_info.refund_blank_id,
            refund_operation_id=ticket_info.refund_operation_id
        )
    return RefundResult(refund_by_blank_id=refund_by_blank_id)
