# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import logging
import operator
import re

from django.utils.encoding import force_text
from django.utils.functional import cached_property
from six import ensure_text

from common.dynamic_settings.default import conf
from common.utils.text import split_string
from travel.proto.dicts.trains.coach_type_pb2 import TCoachType
from travel.proto.dicts.trains.coach_schema_pb2 import TCoachSchema

from travel.rasp.train_api.pb_provider import PROTOBUF_DATA_PROVIDER
from travel.rasp.train_api.train_partners.base.train_details.models import CoachSchemaBinding


log = logging.getLogger(__name__)


class BestSchemaFinder(object):
    def __init__(self, train_number):
        self.bindings = []
        init_method = self._init_with_protobufs if conf.TRAIN_BACKEND_USE_PROTOBUFS['schemas'] else self._init
        init_method(train_number)

    def _init(self, train_number):
        all_bindings = list(CoachSchemaBinding.objects.filter(priority__gt=0).select_related('schema'))
        for binding in all_bindings:
            if not binding.train_number or re.match(binding.train_number, train_number, flags=re.U | re.I):
                self.bindings.append(binding)

    def _init_with_protobufs(self, train_number):
        all_bindings = filter(lambda b: b.Priority > 0, PROTOBUF_DATA_PROVIDER.coach_binding_repo.values())
        for binding in all_bindings:
            if not binding.TrainNumber or re.match(ensure_text(binding.TrainNumber), train_number, flags=re.U | re.I):
                self.bindings.append(binding)

    def find_best_schema(self, coach_type, service_class_code, coach_subtype_code, road, coach_number,
                         two_storey, place_counts, place_numbers, international_service_class_code, im_car_scheme_id):
        find_method = self._find_best_schema_with_protobufs if conf.TRAIN_BACKEND_USE_PROTOBUFS['schemas'] else self._find_best_schema
        return find_method(coach_type, service_class_code, coach_subtype_code, road, coach_number,
                           two_storey, place_counts, place_numbers, international_service_class_code, im_car_scheme_id)

    def _find_best_schema(self, coach_type, service_class_code, coach_subtype_code, road, coach_number,
                          two_storey, place_counts, place_numbers, international_service_class_code, im_car_scheme_id):
        suitable_bindings = []
        for binding in self.bindings:
            if two_storey != binding.schema.two_storey:
                continue
            if binding.klass:
                if coach_type != binding.klass:
                    continue
            if binding.coach_subtype_code:
                if coach_subtype_code not in binding.coach_subtype_codes:
                    continue
            points = 0
            if binding.road:
                if not re.match(binding.road, road, flags=re.U | re.I):
                    continue
                points += 1
            if binding.train_number:
                points += 1
            if binding.service_class:
                if not re.match(binding.service_class, service_class_code, flags=re.U | re.I):
                    continue
                points += 1
            if binding.international_service_class:
                if not re.match(binding.international_service_class, international_service_class_code,
                                flags=re.U | re.I):
                    continue
                points += 1
            if binding.coach_number:
                if not re.match(binding.coach_number, coach_number, flags=re.U | re.I):
                    continue
                points += 1
            if binding.im_car_scheme_id and im_car_scheme_id:
                im_car_scheme_id = str(im_car_scheme_id)
                if not re.match(binding.im_car_scheme_id, im_car_scheme_id, flags=re.U | re.I):
                    continue
                points += 1
            is_valid, error = validate_places(binding.schema, set(place_numbers))
            if not is_valid:
                log.error(
                    'Подобранная схема вагона не валидируется. {error} '
                    'binding=(two_storey={binding.schema.two_storey}, '
                    'klass={binding.klass}, '
                    'coach_subtype_code={coach_subtype_code}, '
                    'road={road}, '
                    'train_number={train_number}, '
                    'service_class={service_class}, '
                    'international_service_class={international_service_class}'
                    'coach_number={binding.coach_number}, '
                    'priority={binding.priority}) '
                    'coach=(coach_type={coach_type}, '
                    'schema={schema}, '
                    'coach.place_counts={place_counts}, '
                    'place_numbers={place_numbers}) '
                    'points={points}'.format(
                        error=error,
                        binding=binding,
                        coach_subtype_code=force_text(binding.coach_subtype_code),
                        road=force_text(binding.road),
                        train_number=force_text(binding.train_number),
                        service_class=force_text(binding.service_class),
                        international_service_class=force_text(binding.international_service_class),
                        coach_type=coach_type,
                        schema=force_text(binding.schema),
                        place_counts=place_counts,
                        place_numbers=set(place_numbers),
                        points=points,
                    )
                )
                continue
            suitable_bindings.append({
                'points': points,
                'priority': binding.priority,
                'schema': binding.schema,
                'binding': binding
            })
        if not suitable_bindings:
            return None, None

        best_binding = max(suitable_bindings, key=operator.itemgetter('priority', 'points'))
        return best_binding['schema'], best_binding['binding']

    def _find_best_schema_with_protobufs(self, coach_type, service_class_code, coach_subtype_code, road, coach_number,
                                         two_storey, place_counts, place_numbers, international_service_class_code,
                                         im_car_scheme_id):
        suitable_bindings = []
        for binding in self.bindings:
            schema = ProtoSchemaProxy(PROTOBUF_DATA_PROVIDER.coach_schema_repo.get(binding.SchemaId))
            if two_storey != schema.two_storey:
                continue
            if binding.Klass:
                if coach_type.upper() != TCoachType.EType.Name(binding.Klass):
                    continue
            if binding.CoachSubtype:
                if coach_subtype_code not in split_string(ensure_text(binding.CoachSubtype)):
                    continue
            points = 0
            if binding.Road:
                if not re.match(ensure_text(binding.Road), road, flags=re.U | re.I):
                    continue
                points += 1
            if binding.TrainNumber:
                points += 1
            if binding.ServiceClass:
                if not re.match(ensure_text(binding.ServiceClass), service_class_code, flags=re.U | re.I):
                    continue
                points += 1
            if binding.InternationalServiceClass:
                if not re.match(ensure_text(binding.InternationalServiceClass), international_service_class_code,
                                flags=re.U | re.I):
                    continue
                points += 1
            if binding.CoachNumber:
                if not re.match(ensure_text(binding.CoachNumber), coach_number, flags=re.U | re.I):
                    continue
                points += 1
            is_valid, error = validate_places(schema, set(place_numbers))
            if not is_valid:
                log.error(
                    'Подобранная схема вагона не валидируется. {error} '
                    'binding=(two_storey={schema.two_storey}, '
                    'klass={binding.Klass}, '
                    'coach_subtype_code={coach_subtype_code}, '
                    'road={road}, '
                    'train_number={train_number}, '
                    'service_class={service_class}, '
                    'international_service_class={international_service_class}'
                    'coach_number={coach_number}, '
                    'priority={binding.Priority}) '
                    'coach=(coach_type={coach_type}, '
                    'schema={schema}, '
                    'coach.place_counts={place_counts}, '
                    'place_numbers={place_numbers}) '
                    'points={points}'.format(
                        error=error,
                        schema=schema,
                        binding=binding,
                        coach_subtype_code=force_text(binding.CoachSubtype),
                        road=force_text(binding.Road),
                        train_number=force_text(binding.TrainNumber),
                        service_class=force_text(binding.ServiceClass),
                        international_service_class=force_text(binding.InternationalServiceClass),
                        coach_number=force_text(binding.CoachNumber),
                        coach_type=coach_type,
                        place_counts=place_counts,
                        place_numbers=set(place_numbers),
                        points=points,
                    )
                )
                continue
            suitable_bindings.append({
                'points': points,
                'priority': binding.Priority,
                'schema': schema,
                'binding': binding
            })
        if not suitable_bindings:
            return None, None

        best_binding = max(suitable_bindings, key=operator.itemgetter('priority', 'points'))
        return best_binding['schema'], best_binding['binding']


def validate_places(schema, free_place_numbers):
    schema_places = {place.number for place in schema.places}

    if free_place_numbers - schema_places:
        return False, 'Среди свободных мест есть номера, которых нет на схеме.'

    return True, ''


class ProtoSchemaProxy(object):
    def __init__(self, proto_schema):
        self.id = proto_schema.Id
        self._places = proto_schema.Places
        self._details = proto_schema.Details
        self.svg_schema = proto_schema.Svg
        self.two_storey = proto_schema.TwoStorey

    @classmethod
    def create(cls, **kwargs):
        return cls(TCoachSchema(**kwargs))

    @cached_property
    def _details_dict(self):
        details_json = self._details.strip()
        if not details_json:
            return {}

        try:
            details_dict = json.loads(details_json)
            if not isinstance(details_dict, dict):
                log.error('JSON CoachSchema.details не являятся словарем, id={}.'.format(self.id))
                return {}
            return details_dict
        except ValueError:
            log.error('Не удалось распарсить JSON CoachSchema.details, id={}.'.format(self.id))
            return {}

    @property
    def place_flags(self):
        return self._details_dict.get('placeFlags', {})

    @property
    def hide_place_numbers(self):
        return self._details_dict.get('hidePlaceNumbers', False)

    @cached_property
    def places(self):
        all_places = map(int, split_string(self._places))
        group_numbers = self.place_flags.get('compartments', [])
        group_number_by_place = {place: group_i for group_i, group in enumerate(group_numbers, 1) for place in group}
        return [self.CoachPlace(place, group_number_by_place.get(place)) for place in all_places]

    class CoachPlace(object):
        def __init__(self, number, group_number):
            self.number = number
            self.group_number = group_number
