# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from collections import namedtuple

from common.utils.date import naive_to_timestamp, UTC_TZ
from travel.rasp.train_api.train_bandit_api.client import BanditContext, BanditClient, CoachData, PlaceData, RequestContext
from travel.rasp.train_api.train_purchase.utils.fee_calculator import TicketCost, calculate_ticket_cost

log = logging.getLogger(__name__)


TicketCostKey = namedtuple(
    'TicketCostKey',
    ('coach_type', 'service_class', 'amount', 'service_amount'),
)


class LazyBanditCostCache(object):
    def __init__(self, icookie, yandex_uid, train_details, bandit_type, in_suburban_search, query):
        self.yandex_uid = yandex_uid
        self.bandit_type = bandit_type
        self.in_suburban_search = in_suburban_search
        self.icookie = icookie
        self.point_from = train_details.station_from.point_key
        self.point_to = train_details.station_to.point_key
        self.departure = naive_to_timestamp(train_details.departure.astimezone(UTC_TZ).replace(tzinfo=None))
        self.arrival = naive_to_timestamp(train_details.arrival.astimezone(UTC_TZ).replace(tzinfo=None))
        self.train_type = train_details.raw_train_name
        self.contract = train_details.contract
        self.log_info = RequestContext(query.get('req_id'), query.get('yandex_uid'), query.get('user_device'))
        self._keys = {}
        self._consumers = []

    def calculate_ticket_cost_lazy(self, coach_type, service_class, amount, service_amount, ticket_cost_consumer_func):
        key = TicketCostKey(coach_type, service_class, amount, service_amount)
        index = self._keys.get(key)
        if index is None:
            index = len(self._consumers)
            self._consumers.append(list())
            self._keys[key] = index
        self._consumers[index].append(ticket_cost_consumer_func)

    def create_context(self, car_type):
        return BanditContext(
            icookie=self.icookie,
            point_from=self.point_from,
            point_to=self.point_to,
            departure=self.departure,
            arrival=self.arrival,
            train_type=self.train_type,
            car_type=car_type,
            in_suburban_search=self.in_suburban_search
        )

    def materialize_ticket_costs(self):
        if not self._keys:
            return
        client = BanditClient(bandit_type=self.bandit_type)
        coach_datas = []
        for key, index in self._keys.items():
            context = self.create_context(key.coach_type)
            place = PlaceData(id=index, amount=key.amount, service_amount=key.service_amount)
            coach_data = CoachData(context=context, places=[place], service_class=key.service_class,
                                   log_info=self.log_info)
            coach_datas.append(coach_data)

        place_fees_by_place_data = None
        try:
            place_fees_by_place_data = client.get_fee_for_coaches(coach_datas)
        except Exception:
            log.exception('Error getting fee from bandit')

        if place_fees_by_place_data:
            for place_data, place_fee in place_fees_by_place_data.items():
                ticket_cost = TicketCost(
                    amount_without_fee=place_data.amount,
                    bedding_amount_without_fee=place_data.service_amount,
                    yandex_fee_percent=place_fee.fee_percent,
                    main_fee=place_fee.main_fee,
                    bedding_fee=place_fee.service_fee,
                    is_bandit_fee_applied=place_fee.is_bandit_fee_applied,
                    bandit_type=place_fee.bandit_type,
                    bandit_version=place_fee.bandit_version,
                )
                for consumer in self._consumers[place_data.id]:
                    consumer(ticket_cost)
        else:
            for key, index in self._keys.items():
                ticket_cost = calculate_ticket_cost(self.contract, key.coach_type, key.amount, key.service_amount, self.yandex_uid)
                for consumer in self._consumers[index]:
                    consumer(ticket_cost)
